package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import java.time.Instant;
import java.util.Map;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.util.DateUtil.timestampToIsoFormat;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;

/**
 * The {@code date} node displays a date in the user's locale.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 * {@link Paragraph#p(InlineContent...) p}(
 *     {@link Text#text(String) text}("I wrote this on "),
 *     {@link #date(long) date}(1690344879000L)
 * );
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "paragraph",
 *    "content": [
 *      {
 *        "type": "text",
 *        "text": "I wrote this on "
 *      },
 *      {
 *        "type": "date",
 *        "attrs": {
 *          "timestamp": "1690344879000"
 *        }
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>I wrote this on
 * <span style="background-color: rgba(9, 30, 66, 0.08); border-collapse: collapse; border-radius: 3px;">26 Jul
 * 2023</span></p>
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the date, so while it gives a vague
 * impression of what a "date" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products. This example also has a hardcoded rendering that does not respect the viewer's
 * locale.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Date
        extends AbstractNode<Date>
        implements CaptionContent, InlineContent {

    static final Factory<Date> FACTORY = new Factory<>(Type.DATE, Date.class, Date::parse);

    private String timestamp;

    private Date(String timestamp) {
        this.timestamp = validateTimestamp(timestamp);
    }

    public static Partial.NeedsTimestamp date() {
        return new Partial.NeedsTimestamp();
    }

    public static Date date(String timestamp) {
        return new Date(timestamp);
    }

    public static Date date(long epochMillis) {
        return date(Instant.ofEpochMilli(epochMillis));
    }

    public static Date date(Instant instant) {
        return new Date(instant.toString());
    }

    public static Date date(java.util.Date date) {
        return date(date.getTime());
    }

    public Date timestamp(String timestamp) {
        this.timestamp = validateTimestamp(timestamp);
        return this;
    }

    public String timestamp() {
        return timestamp;
    }

    @Override
    public String elementType() {
        return Type.DATE;
    }

    @Override
    public Date copy() {
        return parse(toMap());
    }

    @Override
    public void validate() {
    }

    @Override
    protected boolean nodeEquals(Date other) {
        return timestamp.equals(other.timestamp);
    }

    @Override
    protected int nodeHashCode() {
        return timestamp.hashCode();
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendField("timestamp", timestamp);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.TIMESTAMP, timestamp));
    }

    private static Date parse(Map<String, ?> map) {
        checkType(map, Type.DATE);
        String timestamp = getAttrOrThrow(map, Attr.TIMESTAMP);
        return date(timestamp);
    }

    private static String validateTimestamp(String timestamp) {
        //TODO Validate timestamp as ISO 8601? Its intended format is not documented.
        return nonEmpty(timestamp, "timestamp");
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append(timestampToIsoFormat(timestamp));
    }

    /**
     * Types that represent a partially constructed {@link Date}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code date} still needs its {@code timestamp} attribute set.
         */
        class NeedsTimestamp {
            NeedsTimestamp() {
            }

            public Date timestamp(String timestamp) {
                return new Date(timestamp);
            }
        }
    }
}
