package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.Breakout;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.type.CodeBlockMark;
import com.atlassian.adf.model.node.type.*;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttr;

/**
 * A container for lines of code.
 * This is roughly equivalent to the combination of the {@code <pre>} and {@code <code>} HTML element types,
 * although there may be more complicated styling, including language-aware syntax highlighting.
 * <p>
 * If the {@code language} is unspecified or its value is not one of the
 * <a href="https://github.com/react-syntax-highlighter/react-syntax-highlighter/blob/master/AVAILABLE_LANGUAGES_PRISM.MD">supported
 * languages</a>, then the {@code codeBlock} renders as plain, monospace text.
 * <p>
 * Although {@code codeBlock} accepts {@code text} nodes as content, they are not permitted to use any marks.
 * </p>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #codeBlock(String) codeBlock}("var foo = {};&#x5C;nvar bar = [];")
 *         .{@link #language(String) language}("javascript");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "codeBlock",
 *     "attrs": {
 *       "language": "javascript"
 *     },
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "var foo = {};\nvar bar = [];"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <pre>{@code
 *     var foo = {};
 *     var bar = [];
 * }</pre>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/codeBlock/">Node - codeBlock</a>
 * @see <a href="https://github.com/react-syntax-highlighter/react-syntax-highlighter/blob/master/AVAILABLE_LANGUAGES_PRISM.MD">Prism
 * &mdash; Available language imports</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class CodeBlock
        extends AbstractMarkedContentNode<CodeBlock, Text, CodeBlockMark>
        implements DocContent, LayoutColumnContent, ListItemContent, NonNestableBlockContent, TableCellContent {

    static Factory<CodeBlock> FACTORY = new Factory<>(Type.CODE_BLOCK, CodeBlock.class, CodeBlock::parse);

    @Nullable
    private String language;

    private CodeBlock() {
    }

    /**
     * @return a new, empty code block. At least one text node must be added for it to be valid.
     */
    public static CodeBlock codeBlock() {
        return new CodeBlock();
    }

    /**
     * @param content the text to include in the code block. As a convenience, it is implicitly wrapped in a
     *                {@link Text text} node.
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(String content) {
        return codeBlock().content(text(content));
    }

    /**
     * @param content the text to include in the code block. As a convenience, it is implicitly wrapped as
     *                {@link Text text} nodes.
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(String... content) {
        return codeBlock().content(text(content));
    }

    /**
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(Text content) {
        return codeBlock().content(content);
    }

    /**
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(Text... content) {
        return codeBlock().content(content);
    }

    /**
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(Iterable<? extends Text> content) {
        return codeBlock().content(content);
    }

    /**
     * @return a new code block with the given content
     */
    public static CodeBlock codeBlock(Stream<? extends Text> content) {
        return codeBlock().content(content);
    }

    /**
     * @see #codeBlock()
     */
    public static CodeBlock pre() {
        return new CodeBlock();
    }

    /**
     * @see #codeBlock(String)
     */
    public static CodeBlock pre(String content) {
        return pre().content(text(content));
    }

    /**
     * @see #codeBlock(String[])
     */
    public static CodeBlock pre(String... content) {
        return pre().content(text(content));
    }

    /**
     * @see #codeBlock(Text)
     */
    public static CodeBlock pre(Text content) {
        return pre().content(content);
    }

    /**
     * @see #codeBlock(Text[])
     */
    public static CodeBlock pre(Text... content) {
        return pre().content(content);
    }

    /**
     * @see #codeBlock(Iterable)
     */
    public static CodeBlock pre(Iterable<? extends Text> content) {
        return pre().content(content);
    }

    /**
     * @see #codeBlock(Stream)
     */
    public static CodeBlock pre(Stream<? extends Text> content) {
        return pre().content(content);
    }

    @Override
    public CodeBlock copy() {
        return parse(toMap());
    }

    /**
     * @param content the text to include in the code block. As a convenience, it is implicitly wrapped in a
     *                {@link Text text} node.
     * @return {@code this}
     */
    public CodeBlock content(String content) {
        return content(text(content));
    }

    /**
     * @param content the text to include in the code block. As a convenience, it is implicitly wrapped as
     *                {@link Text text} nodes.
     * @return {@code this}
     */
    public CodeBlock content(String... content) {
        return content(text(content));
    }

    /**
     * Returns the language setting for this code block, if set.
     *
     * @return the language setting for this code block, or {@code empty()} if not set.
     */
    public Optional<String> language() {
        return Optional.ofNullable(language);
    }

    /**
     * Sets the language that should be used to interpret the code block's contents.
     *
     * @param language the language code for the table language. Unsupported values result in plain, monospace text.
     * @return {@code this}
     * @see <a href="https://github.com/react-syntax-highlighter/react-syntax-highlighter/blob/master/AVAILABLE_LANGUAGES_PRISM.MD">Available languages</a>
     */
    public CodeBlock language(@Nullable String language) {
        this.language = language;
        return this;
    }

    public Optional<Breakout> breakout() {
        return marks.get(Mark.Type.BREAKOUT)
                .map(Breakout.class::cast);
    }

    public CodeBlock breakout(@Nullable Breakout breakout) {
        marks.clear();
        if (breakout != null) {
            marks.add(breakout);
        }
        return this;
    }

    public CodeBlock wide() {
        return breakout(Breakout.wide());
    }

    public CodeBlock fullWidth() {
        return breakout(Breakout.fullWidth());
    }

    public CodeBlock defaultWidth() {
        return breakout(null);
    }

    @Override
    protected void validateContentNodeForAppend(Text node) {
        super.validateContentNodeForAppend(node);
        node.disableMarks(this);
    }

    @Override
    public String elementType() {
        return Type.CODE_BLOCK;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map().addIfPresent(Attr.LANGUAGE, language))
                .let(this::addContentIfPresent)
                .let(marks::addToMap);
    }

    private static CodeBlock parse(Map<String, ?> map) {
        checkType(map, Type.CODE_BLOCK);
        CodeBlock pre = pre().parseOptionalContent(map, Text.class);
        getAttr(map, Attr.LANGUAGE, String.class).ifPresent(pre::language);
        return pre.parseMarks(map);
    }

    @Override
    public Class<CodeBlockMark> markClass() {
        return CodeBlockMark.class;
    }

    @Override
    protected boolean markedContentNodeEquals(CodeBlock other) {
        return Objects.equals(language, other.language);
    }

    @Override
    protected int markedContentNodeHashCode() {
        return Objects.hashCode(language);
    }

    @Override
    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
        buf.appendField("language", language);
    }

    void disableMarks(ContentNode<?, ? super CodeBlock> parent) {
        marks.disable(parent.elementType());
    }
}
