package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.Functions.iterateJoined;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * A container for {@link ListItem list items} that produces a bulleted list.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #bulletList() bulletList}()
 *         .{@link #li(String) li}("Hello")
 *         .{@link #li(String) li}("World");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "bulletList",
 *     "content": [
 *       {
 *         "type": "listItem",
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "Hello"
 *               }
 *             ]
 *           }
 *         ]
 *       },
 *       {
 *         "type": "listItem",
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "World"
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <ul>
 *     <li><p>Hello</p></li>
 *     <li><p>World</p></li>
 * </ul>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/bulletList/">Node - bulletList</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class BulletList extends AbstractListNode<BulletList> {

    static Factory<BulletList> FACTORY = new Factory<>(Type.BULLET_LIST, BulletList.class, BulletList::parse);

    private BulletList() {
    }

    /**
     * @return a new, empty bulleted list. At least one list item must be added before it will be valid.
     */
    public static BulletList ul() {
        return new BulletList();
    }

    /**
     * @return a new bulleted list with the given contents
     */
    public static BulletList ul(ListItem... content) {
        return ul().content(content);
    }

    /**
     * @return a new bulleted list with the given contents
     */
    public static BulletList ul(Iterable<? extends ListItem> content) {
        return ul().content(content);
    }

    /**
     * @return a new bulleted list with the given contents
     */
    public static BulletList ul(Stream<? extends ListItem> content) {
        return ul().content(content);
    }

    /**
     * @see #ul()
     */
    public static BulletList bulletList() {
        return new BulletList();
    }

    /**
     * @see #ul(ListItem[])
     */
    public static BulletList bulletList(ListItem... content) {
        return bulletList().content(content);
    }

    /**
     * @see #ul(Iterable)
     */
    public static BulletList bulletList(Iterable<? extends ListItem> content) {
        return bulletList().content(content);
    }

    /**
     * @see #ul(Stream)
     */
    public static BulletList bulletList(Stream<? extends ListItem> content) {
        return ul().content(content);
    }

    @Override
    public BulletList copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.BULLET_LIST;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent);
    }

    private static BulletList parse(Map<String, ?> map) {
        checkType(map, Type.BULLET_LIST);
        return ul().parseRequiredContent(map, ListItem.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        if (content.isEmpty()) return;

        sb.append("* ");
        iterateJoined(
                content,
                child -> child.appendPlainText(sb),
                () -> sb.append("\n* ")
        );
    }
}
