package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.BodiedExtensionMark;
import com.atlassian.adf.model.node.ExtensionSettings.Layout;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.ExtensionNode;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.model.node.ExtensionSettings.extensionSettings;
import static com.atlassian.adf.model.node.unsupported.UnsupportedNode.plainTextFallback;
import static com.atlassian.adf.util.ParserSupport.getAttr;

/**
 * Extensions provide hook points for ecosystem add-ons to integrate with how ADF content is rendered.
 * The {@code bodiedExtension} node type is used in contexts that only permit "block" content, such as
 * at the top level (in {@code doc} itself) or in the {@code layoutColumn} nodes that are permitted in
 * Confluence.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class BodiedExtension
        extends AbstractMarkedContentNode<BodiedExtension, NonNestableBlockContent, BodiedExtensionMark>
        implements ExtensionNode<BodiedExtension, BodiedExtensionMark>,
        DocContent, LayoutColumnContent {

    static final Factory<BodiedExtension> FACTORY = new Factory<>(
            Type.BODIED_EXTENSION,
            BodiedExtension.class,
            BodiedExtension::parse
    );

    private final ExtensionSettings settings;

    @Nullable
    private Layout layout;

    private BodiedExtension(ExtensionSettings settings) {
        this.settings = settings;
    }

    @CheckReturnValue
    public static ExtensionSettings.Partial.NeedsExtensionKey<BodiedExtension> bodiedExtension() {
        return new ExtensionSettings.Partial.NeedsExtensionKey<>(BodiedExtension::new);
    }

    public static BodiedExtension bodiedExtension(String extensionKey, String extensionType) {
        return extensionSettings(BodiedExtension::new)
                .extensionKey(extensionKey)
                .extensionType(extensionType);
    }

    @Override
    public BodiedExtension copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.BODIED_EXTENSION;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .add(Key.ATTRS, settings.toExtensionAttrs()
                        .addMappedIfPresent(Attr.LAYOUT, layout, Layout::layout)
                )
                .let(this::addContent)
                .let(marks::addToMap);
    }

    @Override
    protected void validateContentNodeForAppend(NonNestableBlockContent node) {
        super.validateContentNodeForAppend(node);
        if (node instanceof Paragraph) {
            ((Paragraph) node).disableMarks(this);
        } else if (node instanceof Heading) {
            ((Heading) node).disableMarks(this);
        } else if (node instanceof CodeBlock) {
            ((CodeBlock) node).disableMarks(this);
        }
    }

    @Override
    public Class<BodiedExtensionMark> markClass() {
        return BodiedExtensionMark.class;
    }

    @Override
    protected void markedContentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    protected int markedContentNodeHashCode() {
        return Objects.hash(layout, settings);
    }

    @Override
    protected boolean markedContentNodeEquals(BodiedExtension other) {
        return layout == other.layout
                && settings.equals(other.settings);
    }

    @Override
    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
        settings.appendExtensionFields(buf);
        buf.appendField("layout", layout);
    }

    @Override
    public String extensionKey() {
        return settings.extensionKey();
    }

    @Override
    public BodiedExtension extensionKey(String extensionKey) {
        settings.extensionKey(extensionKey);
        return this;
    }

    @Override
    public String extensionType() {
        return settings.extensionType();
    }

    @Override
    public BodiedExtension extensionType(String extensionType) {
        settings.extensionType(extensionType);
        return this;
    }

    @Override
    public BodiedExtension localId(@Nullable String localId) {
        settings.localId(localId);
        return this;
    }

    @Override
    public Optional<String> localId() {
        return settings.localId();
    }

    @Override
    public BodiedExtension text(@Nullable String text) {
        settings.text(text);
        return this;
    }

    @Override
    public Optional<String> text() {
        return settings.text();
    }

    @Override
    public BodiedExtension parameters(@Nullable Map<String, ?> parameters) {
        settings.parameters(parameters);
        return this;
    }

    @Override
    public Optional<Map<String, ?>> parameters() {
        return settings.parameters();
    }

    public BodiedExtension layout(String layout) {
        return layout(Layout.PARSER.parse(layout));
    }

    public BodiedExtension layout(Layout layout) {
        this.layout = layout;
        return this;
    }

    public Optional<Layout> layout() {
        return Optional.ofNullable(layout);
    }

    private static BodiedExtension parse(Map<String, ?> map) {
        ExtensionSettings settings = ExtensionSettings.parse(map);
        BodiedExtension extension = new BodiedExtension(settings)
                .parseRequiredContent(map, NonNestableBlockContent.class);
        getAttr(map, Attr.LAYOUT, String.class).ifPresent(layout -> extension.layout(Layout.PARSER.parse(layout)));
        return extension.parseMarks(map);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String text = settings.text().orElseGet(() -> plainTextFallback(this));
        sb.append(text);
    }
}
