package com.atlassian.adf.model.node;

import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.model.node.Paragraph.p;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code Blockquote} node is a container for quotes.
 * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content}.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #bq(Paragraph[]) bq}(
 *         {@link Paragraph#p(String) p}("Hello world"),
 *         {@link Paragraph#p(String) p}("&#x5C;u2014 Atlassian Document Format")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "blockquote",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello world"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "\u2014 Atlassian Document Format"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <blockquote>
 * <p>Hello, world</p>
 * <p>&mdash; Atlassian Document Format</p>
 * </blockquote>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/blockquote/">Node - blockquote</a>
 */
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class Blockquote
        extends AbstractContentNode<Blockquote, Paragraph>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent {

    static Factory<Blockquote> FACTORY = new Factory<>(Type.BLOCKQUOTE, Blockquote.class, Blockquote::parse);

    private Blockquote() {
    }

    /**
     * Create an empty {@code Blockquote} node.
     * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content} to make it valid.
     *
     * @return a new, empty {@code Blockquote}
     */
    public static Blockquote bq() {
        return new Blockquote();
    }

    /**
     * Create a {@code Blockquote} node containing a single paragraph with the given text as its content.
     *
     * @param content the string to use as the content of the block quote. As a convenience, the provided
     *                string is wrapped first as a {@link Text} node and then as a {@link Paragraph} before
     *                it is set as the content. It cannot be an empty string.
     * @return a new {@code Blockquote} containing the provided text as its content
     */
    public static Blockquote bq(String content) {
        return bq().content(p(content));
    }

    /**
     * Create a {@code Blockquote} node containing a single paragraph with the given text as its content.
     * As a convenience, each element of {@code content} is wrapped first as a {@link Text} node and then
     * as a {@link Paragraph} before it is set as the content for this {@code Blockquote}.
     * <p>
     * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content} to make it valid.
     *
     * @param content the strings to use as the content of the block quote. As a convenience, each provided
     *                string is wrapped first as a {@link Text} node and then as a {@link Paragraph} before
     *                it is set as the content, so each element will form its own paragraph. None of them may
     *                be an empty string.
     * @return a new {@code Blockquote} containing the provided text as its content
     */
    public static Blockquote bq(String... content) {
        return bq().content(Arrays.stream(content)
                .map(Text::text)
                .map(Paragraph::p));
    }

    /**
     * Create a {@code Blockquote} node containing a single paragraph as its content.
     *
     * @param content the paragraph to use as the content of the block quote.
     * @return a new {@code Blockquote} containing the provided paragraph as its content
     */
    public static Blockquote bq(Paragraph content) {
        return bq().content(content);
    }

    /**
     * Create a {@code Blockquote} node containing the given paragraph as its content.
     * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content} to make it valid.
     *
     * @param content the paragraphs to use as the content of the block quote.
     * @return a new {@code Blockquote} containing the provided paragraph as its content
     */
    public static Blockquote bq(Paragraph... content) {
        return bq().content(content);
    }

    /**
     * Create a {@code Blockquote} node containing the given paragraph as its content.
     * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content} to make it valid.
     *
     * @param content the paragraphs to use as the content of the block quote. The iterable will be iterated
     *                at most once and immediately.
     * @return a new {@code Blockquote} containing the provided paragraph as its content
     */
    public static Blockquote bq(Iterable<? extends Paragraph> content) {
        return bq().content(content);
    }

    /**
     * Create a {@code Blockquote} node containing the given paragraph as its content.
     * At least one {@link Paragraph} must be provided as the block's {@link AbstractContentNode content} to make it valid.
     *
     * @param content the paragraphs to use as the content of the block quote. The stream will be terminated and
     *                consumed immediately.
     * @return a new {@code Blockquote} containing the provided paragraph as its content
     */
    public static Blockquote bq(Stream<? extends Paragraph> content) {
        return bq().content(content);
    }

    /**
     * @see #bq()
     */
    public static Blockquote blockquote() {
        return new Blockquote();
    }

    /**
     * @see #bq(String)
     */
    public static Blockquote blockquote(String content) {
        return bq(content);
    }

    /**
     * @see #bq(String[])
     */
    public static Blockquote blockquote(String... content) {
        return bq(content);
    }

    /**
     * @see #bq(Paragraph)
     */
    public static Blockquote blockquote(Paragraph content) {
        return bq(content);
    }

    /**
     * @see #bq(Paragraph[])
     */
    public static Blockquote blockquote(Paragraph... content) {
        return bq(content);
    }

    /**
     * @see #bq(Iterable)
     */
    public static Blockquote blockquote(Iterable<? extends Paragraph> content) {
        return bq(content);
    }

    /**
     * @see #bq(Stream)
     */
    public static Blockquote blockquote(Stream<? extends Paragraph> content) {
        return bq(content);
    }

    @Override
    public Blockquote copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.BLOCKQUOTE;
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    protected void validateContentNodeForAppend(Paragraph node) {
        node.disableMarks(this);
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent);
    }

    private static Blockquote parse(Map<String, ?> map) {
        checkType(map, Type.BLOCKQUOTE);
        return bq().parseRequiredContent(map, Paragraph.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append("> ");
        content.forEach(child -> child.appendPlainText(sb));
    }
}
