package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.*;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import java.net.URI;
import java.net.URL;
import java.util.Map;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code blockCard} node is an Atlassian link card with a type icon and content description derived from the
 * link. It provides the same functionality that {@link InlineCard inlineCard} does, but renders its link as a
 * standalone block rather than in combination with the other inline content in a {@link Paragraph paragraph}
 * or similar container.
 * <p>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * {@link Doc#doc(DocContent[] doc) doc}(
 *     {@link Paragraph#p(String) p}("Hello"),
 *     {@link #blockCard(String) blockCard}("https://jira.atlassian.com/browse/JRACLOUD-72220"),
 *     {@link Paragraph#p(String) p}("World")
 * );
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "doc",
 *     "version": 1,
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "blockCard",
 *         "attrs": {
 *           "url": "https://jira.atlassian.com/browse/JRACLOUD-72220"
 *         }
 *       },
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "World"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello</p>
 * <p>
 * <a href="https://jira.atlassian.com/browse/JRACLOUD-72220"><img height="16px" width="16px"
 * src="data:image/png;base64,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" />
 * JRACLOUD-72220</a> -
 * <span class="summary">Java client for building/parsing ADF</span>
 * <span style="border-collapse: collapse; box-sizing: border-box; background-color: #ff991f; color: #rgb(51, 51, 51);">IN PROGRESS</span>
 * </p>
 * <p>World</p>
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the block card, so while it gives a vague
 * impression of what a "block card" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products.
 *
 * @see InlineCard
 */
@SuppressWarnings("JavadocLinkAsPlainText")
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class BlockCard
        extends AbstractCardNode<BlockCard>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, PanelContent, TableCellContent {

    static Factory<BlockCard> FACTORY = new Factory<>(Type.BLOCK_CARD, BlockCard.class, BlockCard::parse);

    private BlockCard(UrlOrData urlOrData) {
        super(urlOrData);
    }

    /**
     * Creates a partially constructed block card that requires its URI or data to be set before it is valid.
     *
     * @return a partially constructed block card
     */
    @CheckReturnValue
    public static Partial.NeedsUrlOrData<BlockCard> blockCard() {
        return new Partial.NeedsUrlOrData<>(BlockCard::new);
    }

    /**
     * Creates a block card for the linked content with the given URI.
     *
     * @param url the target URI for the block card
     * @return a new block card for the given URI
     * @throws IllegalArgumentException if the URI cannot be parsed
     */
    public static BlockCard blockCard(String url) {
        return blockCard().url(url);
    }

    /**
     * Creates a block card for the linked content with the given URI.
     *
     * @param url the target URI for the block card
     * @return a new block card for the given URI
     */
    public static BlockCard blockCard(URL url) {
        return blockCard().url(url);
    }

    /**
     * Creates a block card for the linked content with the given URI.
     *
     * @param url the target URI for the block card
     * @return a new block card for the given URI
     */
    public static BlockCard blockCard(URI url) {
        return blockCard().url(url);
    }

    /**
     * Creates a block card for the linked content with the given JSON-LD description.
     *
     * @param data the link description data in <a href="https://json-ld.org/">JSON-LD</a> format
     * @return a new block card for the given JSON-LD
     */
    public static BlockCard blockCard(Map<String, ?> data) {
        return blockCard().data(data);
    }

    @Override
    public BlockCard copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.BLOCK_CARD;
    }

    private static BlockCard parse(Map<String, ?> map) {
        checkType(map, Type.BLOCK_CARD);
        UrlOrData urlOrData = parseUrlOrData(map);
        return new BlockCard(urlOrData);
    }
}