package com.atlassian.adf.model.node;

import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.util.FieldMap;
import com.atlassian.annotations.Internal;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.atlassian.adf.model.node.unsupported.UnsupportedNode.appendPlainTextForUnsupportedNode;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;

@Internal
public abstract class AbstractNode<N extends AbstractNode<N>> implements Node {
    static final int TEXT_BUFFER_SIZE = 8192;

    private static final double EPSILON = 1e-7;
    private static final double ONE_OVER_EPSILON = 1e7;

    protected FieldMap mapWithType() {
        return map(Key.TYPE, elementType());
    }

    @Override
    public final boolean isSupported() {
        return true;
    }

    /**
     * Hashes the node's class together with {@link #nodeHashCode()}.
     * Implementations that define their own private fields should override {@code nodeHashCode()} to include
     * their information in the hashed value.
     *
     * @see #nodeHashCode()
     */
    @Override
    public final int hashCode() {
        return getClass().hashCode() * 31 + nodeHashCode();
    }

    /**
     * After checking for the trivial cases of {@code obj == this}, {@code obj == null}, or
     * {@code getClass() != obj.getClass()}, delegates equality testing to {@link #nodeEquals(AbstractNode)} .
     *
     * @see #nodeEquals(AbstractNode)
     */
    @Override
    public final boolean equals(@Nullable Object obj) {
        if (obj == this) return true;
        if (obj == null || obj.getClass() != getClass()) return false;
        N other = unsafeCast(obj);
        return nodeEquals(other);
    }

    /**
     * Generates a {@code toString()} that is suitable for debugging or logging.
     * <p>
     * Note that this is not the same as the {@link #toPlainText() plain-text} rendering of the node,
     * which is less informative and in many cases will be entirely empty.
     *
     * @see #toPlainText()
     */
    @Override
    public final String toString() {
        ToStringHelper buf = new ToStringHelper();
        buf.appendNode(this);
        return buf.toString();
    }

    /**
     * Renders this node as a plain-text string representation that is suitable for viewing by end users.
     * <p>
     * Note that this is not the same as the standard Java {@link Object#toString()} method, which
     * this library reserves for debugging and logging use, only.
     */
    @Override
    public final String toPlainText() {
        var sb = new StringBuilder(TEXT_BUFFER_SIZE);
        appendPlainText(sb);
        return sb.toString();
    }

    /**
     * Allows nodes that have their own fields to augment the {@code hashCode} implementation with
     * a hash of their own field values.
     * <p>
     * Implementations need not include the node's class; that is already covered by the
     * {@link AbstractNode#hashCode()} implementation that is expected to be this method's
     * only consumer.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractNode} should maintain consistent implementations
     * of {@code nodeHashCode}, {@code nodeEquals}, and {@code appendNodeFields}.
     *
     * @return the hash code of any additional field values that belong to a particular type of content node.
     * @see #nodeEquals(AbstractNode)
     * @see #appendNodeFields(ToStringHelper)
     */
    protected int nodeHashCode() {
        return 0;
    }

    /**
     * Allows nodes that have their own fields to augment the {@code equals} implementation with
     * tests for their own field values.
     * <p>
     * Implementations need not check for identity, {@code null}, or a different node class; those are
     * already covered by the {@link AbstractNode#equals(Object)} implementation that is expected to
     * be this method's only consumer.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractNode} should maintain consistent implementations
     * of {@code nodeHashCode}, {@code nodeEquals}, and {@code appendNodeFields}.
     *
     * @return {@code true} if all additional field values that belong to a particular type of content node
     * test as equal; {@code false} if differences are found
     * @see #nodeHashCode()
     * @see #appendNodeFields(ToStringHelper)
     */
    protected boolean nodeEquals(N other) {
        return true;
    }

    /**
     * Allows nodes that have their own fields to augment the {@code toString()} implementation with
     * their own field values.
     * <p>
     * Each field's value should be provided by calling {@link ToStringHelper#appendField(String, Object)}.
     * The {@code value} may be {@code null}, in which case the field is omitted, for brevity. It will
     * handle array values gracefully, including arrays of primitive types.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractNode} should maintain consistent implementations
     * of {@code nodeHashCode}, {@code nodeEquals}, and {@code appendNodeFields}.
     *
     * @param buf where the field values should be written
     * @see #nodeHashCode()
     * @see #nodeEquals(AbstractNode)
     */
    protected void appendNodeFields(ToStringHelper buf) {
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        if (this instanceof ContentNode<?, ?>) {
            // Paranoid. Since AbstractContentNode overrides this method, it seems unlikely that this code
            // would ever get reached. However, since the typescript code doesn't have the corresponding
            // abstract classes in its hierarchy, it has to rely on checking for content at this point,
            // instead, and that's how it deals with the case of things that are content nodes but don't
            // have explicit mappings. This is just a direct translation of that.
            ContentNode<?, ?> contentNode = unsafeCast(this);
            AbstractContentNode.appendPlainTextContent(sb, contentNode);
        } else {
            appendPlainTextForUnsupportedNode(this, sb);
        }
    }

    @SuppressWarnings("unchecked")
    protected final N self() {
        return (N) this;
    }

    protected static boolean doubleEq(double a, double b) {
        return Double.doubleToLongBits(a) == Double.doubleToLongBits(b)
                || Math.abs(a - b) <= EPSILON;
    }

    protected static boolean numberEq(@Nullable Number a, @Nullable Number b) {
        return (a != null)
                ? b != null && doubleEq(a.doubleValue(), b.doubleValue())
                : b == null;
    }

    protected static int doubleHash(double value) {
        return Long.hashCode((long) (value * ONE_OVER_EPSILON));
    }

    protected static int numberHash(@Nullable Number value) {
        return (value != null) ? doubleHash(value.doubleValue()) : 0;
    }

    protected static class ToStringHelper {
        private final StringBuilder sb;

        ToStringHelper() {
            sb = new StringBuilder(TEXT_BUFFER_SIZE);
        }

        void appendField(String name, @Nullable Object value) {
            if (value == null) return;
            sb.append(name).append('=');
            appendValue(value);
            comma();
        }

        void appendTextField(@Nullable String text) {
            if (text == null) return;
            appendString(text);
            comma();
        }

        void appendContentField(List<? extends Node> content) {
            if (content.isEmpty()) return;
            sb.append("content=");
            appendIterable(content);
            comma();
        }

        void appendMarksField(MarkHolder<? extends Mark> marks) {
            if (marks.isEmpty()) return;
            sb.append("marks=");
            appendIterable(marks.get());
            comma();
        }

        private void appendValue(@Nullable Object value) {
            if (value == null) {
                sb.append("null");
            } else if (value instanceof String) {
                appendString((String) value);
            } else if (value instanceof AbstractNode<?>) {
                appendNode((AbstractNode<?>) value);
            } else if (value instanceof Iterable<?>) {
                appendIterable((Iterable<?>) value);
            } else if (value instanceof Object[]) {
                appendObjectArray((Object[]) value);
            } else if (value.getClass().isArray()) {
                sb.append(primitiveArrayToString(value));
            } else if (value instanceof Map<?, ?>) {
                appendMap((Map<?, ?>) value);
            } else {
                sb.append(value);
            }
        }

        // This uses guillamets (doubled sideways chevrons) to quote strings in a way that's less likely
        // to collide with the actual data.
        @SuppressWarnings("UnnecessaryUnicodeEscape")
        private void appendString(String value) {
            sb.append('\u00AB')
                    .append(value)
                    .append('\u00BB');
        }

        private void appendNode(AbstractNode<?> node) {
            sb.append(node.elementType()).append('[');
            node.appendNodeFields(this);
            int pos = sb.length() - 2;
            if (pos > 0 && sb.charAt(pos) == ',' && sb.charAt(pos + 1) == ' ') sb.setLength(pos);
            sb.append(']');
        }

        private void appendObjectArray(Object[] value) {
            sb.append('[');
            for (Object item : value) {
                appendValue(item);
                comma();
            }
            dropLastComma();
            sb.append(']');
        }

        private void appendIterable(Iterable<?> value) {
            sb.append('[');
            for (Object item : value) {
                appendValue(item);
                comma();
            }
            dropLastComma();
            sb.append(']');
        }

        private void appendMap(Map<?, ?> value) {
            sb.append('{');
            value.forEach((k, v) -> {
                sb.append(k).append(": ");
                appendValue(v);
                comma();
            });
            dropLastComma();
            sb.append('}');
        }

        private void comma() {
            sb.append(", ");
        }

        private void dropLastComma() {
            int pos = sb.length() - 2;
            if (pos > 0 && sb.charAt(pos) == ',' && sb.charAt(pos + 1) == ' ') {
                sb.setLength(pos);
            }
        }

        @Override
        public String toString() {
            return sb.toString();
        }

        private static String primitiveArrayToString(Object value) {
            if (value instanceof int[]) return Arrays.toString((int[]) value);
            if (value instanceof double[]) return Arrays.toString((double[]) value);
            if (value instanceof long[]) return Arrays.toString((long[]) value);
            if (value instanceof char[]) return Arrays.toString((char[]) value);
            if (value instanceof byte[]) return Arrays.toString((byte[]) value);
            if (value instanceof short[]) return Arrays.toString((short[]) value);
            if (value instanceof float[]) return Arrays.toString((float[]) value);
            if (value instanceof boolean[]) return Arrays.toString((boolean[]) value);
            return value.getClass().getSimpleName();
        }
    }
}
