package com.atlassian.adf.model.node;

import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.MarkParserSupport;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.Marked;
import com.atlassian.adf.util.Factory;
import com.atlassian.annotations.Internal;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;

// Java doesn't have multiple inheritance. AbstractMarked is easier to reimplement than AbstractContentNode,
// so the latter wins.
@Internal
public abstract class AbstractMarkedContentNode<
        C extends AbstractMarkedContentNode<C, N, M>,
        N extends Node,
        M extends Mark
        >
        extends AbstractContentNode<C, N>
        implements ContentNode<C, N>, Marked<C, M> {

    protected final MarkHolder<M> marks;

    AbstractMarkedContentNode() {
        this.marks = requireNonNull(createMarkHolder(), "createMarkHolder()");
    }

    @Override
    public abstract Class<M> markClass();

    protected MarkHolder<M> createMarkHolder() {
        return MarkHolder.unlimited();
    }

    @Override
    public final Collection<M> marks() {
        return marks.get();
    }

    @Override
    public final Set<String> markTypes() {
        return marks.getTypes();
    }

    @Override
    public final <T extends M> Stream<? extends T> marks(Class<T> markClass) {
        return marks.stream(markClass);
    }

    @Override
    public final Optional<M> mark(String markType) {
        return marks.get(markType);
    }

    @Override
    public C mark(M mark) {
        marks.add(mark);
        return self();
    }


    protected C parseMarks(Map<String, ?> map) {
        MarkParserSupport.parseMarks(map, markClass(), unsupportedMarkFactory(), this);
        return self();
    }

    @Override
    protected final void contentNodeValidate() {
        marks.validate();
        markedContentNodeValidate();
    }

    protected void markedContentNodeValidate() {
    }

    @Nullable
    protected Factory<M> unsupportedMarkFactory() {
        return null;
    }

    // These methods repeat the delegation pattern established by AbstractNode
    @Override
    protected final int contentNodeHashCode() {
        return marks.hashCode() * 31 + markedContentNodeHashCode();
    }

    @Override
    protected final boolean contentNodeEquals(C other) {
        return markedContentNodeEquals(other)
                && marks.equals(other.marks);
    }

    @Override
    protected final void appendContentNodeFields(ToStringHelper buf) {
        appendMarkedContentNodeFields(buf);
        buf.appendMarksField(marks);
    }

    protected int markedContentNodeHashCode() {
        return 0;
    }

    protected boolean markedContentNodeEquals(C other) {
        return true;
    }

    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
    }
}
