package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code em} mark sets <span style="text-decoration:line-through;">strike-through</span> styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#strike() strike}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "strike"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <span style="text-decoration:line-through;">world</span>!</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/strike/">Mark - strike</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@Immutable
public class Strike
        extends AbstractMark
        implements FormattedTextMark {

    private static final Strike STRIKE = new Strike();
    static final Factory<Strike> FACTORY = new Factory<>(Type.STRIKE, Strike.class, Strike::parse);

    private Strike() {
        // singleton
    }

    /**
     * @return a {@code strike} mark
     */
    public static Strike strike() {
        return STRIKE;
    }

    /**
     * Convenience method for constructing a {@code text} node with a {@code strike} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text strike(String text) {
        return text(text).strike();
    }

    /**
     * Convenience method for applying a {@code strike} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text strike(Text text) {
        return text.strike();
    }

    @Override
    public Strike copy() {
        return this;
    }

    @Override
    public String elementType() {
        return Type.STRIKE;
    }

    private static Strike parse(Map<String, ?> map) {
        checkType(map, Type.STRIKE);
        return strike();
    }
}
