package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.ex.mark.MarkException;
import com.atlassian.adf.model.mark.type.TextMark;
import com.atlassian.adf.model.mark.unsupported.UnsupportedTextMark;
import com.atlassian.adf.model.node.Node;
import com.atlassian.adf.model.node.type.Marked;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;

import static com.atlassian.adf.model.ex.AdfException.MARKS_PREFIX;
import static com.atlassian.adf.model.ex.AdfException.MARK_PREFIX;
import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.ParserSupport.getTypeOrThrow;

/**
 * Used by {@link com.atlassian.adf.model.node.type.Marked marked} nodes to process any {@link Mark marks}
 * that are present.
 *
 * @see #parseMarks(Map, Class, Factory, Marked)
 */
public abstract class MarkParserSupport {
    private static final Map<String, Factory<? extends Mark>> MARK_MAP = Factory.extractLookupMap(
            Factory::type,
            Alignment.FACTORY,
            Annotation.FACTORY,
            Border.FACTORY,
            Breakout.FACTORY,
            Code.FACTORY,
            DataConsumer.FACTORY,
            Em.FACTORY,
            Fragment.FACTORY,
            Indentation.FACTORY,
            Link.FACTORY,
            Strike.FACTORY,
            Strong.FACTORY,
            SubSup.FACTORY,
            TextColor.FACTORY,
            Underline.FACTORY
    );

    private MarkParserSupport() {
        // static-only class
    }

    /**
     * Given a map representation of an object that is expected to represent a mark on a text node,
     * attempts to parse its contents into a valid {@link Mark}.
     *
     * @param map                    the map representation of the mark
     * @param expectedMarkClass      the type of marks accepted
     * @param unsupportedMarkFactory how to make unsupported marks of the appropriate type, or {@code null}
     *                               to reject them by throwing an exception
     * @return the parsed mark
     * @throws MarkException.TypeMismatch    if the mark is successfully parsed but is not an instance of the
     *                                       {@code expectedMarkClass}
     * @throws MarkException.TypeUnsupported if the mark is not recognized as a supported mark type at all
     *                                       and no factory was provided for generating a placeholder for it
     */
    private static <M extends Mark> M parseMark(
            Map<String, ?> map,
            Class<M> expectedMarkClass,
            @Nullable Factory<M> unsupportedMarkFactory
    ) {
        String type = getTypeOrThrow(map);
        return frame(MARK_PREFIX + type, () -> {
            Factory<? extends Mark> factory = MARK_MAP.getOrDefault(type, unsupportedMarkFactory);
            if (factory == null) {
                throw new MarkException.TypeUnsupported(expectedMarkClass, type);
            }
            Mark mark = factory.parse(map);
            if (!expectedMarkClass.isInstance(mark)) {
                throw new MarkException.TypeMismatch(expectedMarkClass, type);
            }
            return expectedMarkClass.cast(mark);
        });
    }

    /**
     * Given a map representation of a marked node, attempts to parse any marks that are present in it.
     *
     * @param nodeMap                the node's map form
     * @param expectedMarkClass      the type of marks accepted by this node
     * @param unsupportedMarkFactory how to make unsupported marks of the appropriate type, or {@code null}
     *                               to reject them by throwing an exception
     * @param markedNode             where to put the marks that we find
     */
    public static <M extends Mark> void parseMarks(
            Map<String, ?> nodeMap,
            Class<M> expectedMarkClass,
            @Nullable Factory<M> unsupportedMarkFactory,
            Marked<?, M> markedNode
    ) {
        List<Map<String, ?>> markList = unsafeCast(nodeMap.get(Node.Key.MARKS));
        if (markList != null) {
            for (int i = 0; i < markList.size(); ++i) {
                Map<String, ?> markMap = markList.get(i);
                frame(MARKS_PREFIX + '[' + i + ']', () -> {
                    M mark = parseMark(markMap, expectedMarkClass, unsupportedMarkFactory);
                    markedNode.mark(mark);
                    return null;
                });
            }
        }
    }

    public static void parseTextMarks(Map<String, ?> map, Marked<?, TextMark> node) {
        parseMarks(map, TextMark.class, UnsupportedTextMark.FACTORY, node);
    }
}
