package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.*;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.cleanUri;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * The {@code link} mark sets a hyperlink.
 * This mark applies to {@link Text} nodes and cannot be combined with the {@link TextColor textColor} mark.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Text#text(String) text}("Hello world").{@link Text#mark(TextMark) mark}(
 *         {@link Link#link(String) link}("http://atlassian.com")
 *                 .{@link Link#title(String) title}("Atlassian")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>
 * {@code
 *   {
 *     "type": "text",
 *     "text": "Hello world",
 *     "marks": [
 *       {
 *         "type": "link",
 *         "attrs": {
 *           "href": "http://atlassian.com",
 *           "title": "Atlassian"
 *         }
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p><a href="http://atlassian.com" title="Atlassian">Hello world</a></p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/link/">Mark - link</a>
 */
@Documentation(
        state = Documentation.State.INCOMPLETE,
        date = "2023-07-26",
        comment = "missing descriptions for media attributes"
)
public class Link
        extends AbstractMark
        implements CodeTextMark, FormattedTextMark, MediaMark, MediaInlineMark, MediaSingleMark {

    static final Factory<Link> FACTORY = new Factory<>(Type.LINK, Link.class, Link::parse);

    private String href;

    @Nullable
    private String title;

    @Nullable
    private String collection;

    @Nullable
    private String id;

    @Nullable
    private String occurrenceKey;

    private Link(String href) {
        this.href = requireNonNull(href, "url");
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public static Link a(String href) {
        return link(href);
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public static Link a(URL href) {
        return link(href);
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     */
    public static Link a(URI href) {
        return link(href);
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public static Link link(String href) {
        return new Link(cleanUri(href, "href"));
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public static Link link(URL href) {
        return new Link(cleanUri(href, "href"));
    }

    /**
     * @param href the URI for the hyperlink, equivalent to the {@code href} value for an HTML {@code <a>} element
     * @return a {@code link} mark
     */
    public static Link link(URI href) {
        return new Link(cleanUri(href, "href"));
    }

    @Override
    public Link copy() {
        return parse(toMap());
    }

    /**
     * Returns the {@code title} value for this link, if set.
     *
     * @return the {@code title} value for this link, or {@code empty()} if not set.
     */
    public Optional<String> title() {
        return Optional.ofNullable(title);
    }

    /**
     * Returns the {@code href} value for the link; that is, its target URI.
     */
    public String href() {
        return href;
    }

    /**
     * Modifies this link's target.
     *
     * @param href the new target URI for this link
     * @return {@code this}
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public Link href(String href) {
        this.href = cleanUri(href, "href");
        return this;
    }

    /**
     * Modifies this link's target.
     *
     * @param href the new target URI for this link
     * @return {@code this}
     * @throws AdfException.InvalidURI if {@code href} cannot be parsed as a valid URI
     */
    public Link href(URL href) {
        this.href = cleanUri(href, "href");
        return this;
    }

    /**
     * Modifies this link's target.
     *
     * @param href the new target URI for this link
     * @return {@code this}
     */
    public Link href(URI href) {
        this.href = cleanUri(href, "href");
        return this;
    }

    /**
     * @param title the title text for the hyperlink, equivalent to the {@code title} value for an HTML
     *              {@code <a>} element
     * @return {@code this}
     */
    public Link title(@Nullable String title) {
        this.title = title;
        return this;
    }

    // TODO collection, id, and occurrenceKey have no description in the ADF docs. Same values as for media?
    public Optional<String> collection() {
        return Optional.ofNullable(collection);
    }

    public Link collection(@Nullable String collection) {
        this.collection = collection;
        return this;
    }

    public Optional<String> id() {
        return Optional.ofNullable(id);
    }

    public Link id(String id) {
        this.id = id;
        return this;
    }

    public Optional<String> occurrenceKey() {
        return Optional.ofNullable(occurrenceKey);
    }

    public Link occurrenceKey(String occurrenceKey) {
        this.occurrenceKey = occurrenceKey;
        return this;
    }

    @Override
    public String elementType() {
        return Type.LINK;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.HREF, href)
                        .addIfPresent(Attr.TITLE, title)
                        .addIfPresent(Attr.COLLECTION, collection)
                        .addIfPresent(Attr.ID, id)
                        .addIfPresent(Attr.OCCURRENCE_KEY, occurrenceKey)
                );
    }

    private static Link parse(Map<String, ?> map) {
        checkType(map, Type.LINK);
        Link link = new Link(getAttrOrThrow(map, Attr.HREF, String.class));
        getAttr(map, Attr.TITLE, String.class).ifPresent(link::title);
        getAttr(map, Attr.COLLECTION, String.class).ifPresent(link::collection);
        getAttr(map, Attr.ID, String.class).ifPresent(link::id);
        getAttr(map, Attr.OCCURRENCE_KEY, String.class).ifPresent(link::occurrenceKey);
        return link;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Link link = (Link) o;
        return href.equals(link.href)
                && Objects.equals(title, link.title)
                && Objects.equals(id, link.id)
                && Objects.equals(collection, link.collection)
                && Objects.equals(occurrenceKey, link.occurrenceKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(href, title, id, collection, occurrenceKey);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(elementType())
                .append("[href=").append(href);
        if (title != null) sb.append(", title=").append(title);
        if (id != null) sb.append(", id=").append(id);
        if (collection != null) sb.append(", collection=").append(collection);
        if (occurrenceKey != null) sb.append(", occurrenceKey=").append(occurrenceKey);
        return sb.append(']').toString();
    }
}
