package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.BodiedExtensionMark;
import com.atlassian.adf.model.mark.type.ExtensionMark;
import com.atlassian.adf.model.mark.type.InlineExtensionMark;
import com.atlassian.adf.model.mark.type.TableMark;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

/**
 * Its purpose seems to be related to giving extension nodes a way to refer to page fragments that should be
 * integrated into the rendering at this point.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Fragment
        extends AbstractMark
        implements BodiedExtensionMark, ExtensionMark, InlineExtensionMark, TableMark {

    static final Factory<Fragment> FACTORY = new Factory<>(Type.FRAGMENT, Fragment.class, Fragment::parse);

    private final String localId;

    @Nullable
    private String name;

    private Fragment(String localId) {
        this.localId = requireNonNull(localId, "localId");
    }

    public static Partial.NeedsLocalId fragment() {
        return Partial.NeedsLocalId.INSTANCE;
    }

    public static Fragment fragment(String localId) {
        return new Fragment(localId);
    }

    @Override
    public Fragment copy() {
        return parse(toMap());
    }

    public Fragment name(@Nullable String name) {
        this.name = name;
        return this;
    }

    @Override
    public String elementType() {
        return Type.FRAGMENT;
    }

    public String localId() {
        return localId;
    }

    public Optional<String> name() {
        return Optional.ofNullable(name);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.LOCAL_ID, localId)
                        .addIfPresent(Attr.NAME, name)
                );
    }

    @Override
    public boolean equals(@Nullable Object o) {
        return this == o || (o instanceof Fragment && equalTo((Fragment) o));
    }

    private boolean equalTo(Fragment other) {
        return localId.equals(other.localId)
                && Objects.equals(name, other.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(localId, name);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(elementType())
                .append("[localId=").append(localId);
        if (name != null) sb.append(", name=").append(name);
        return sb.append(']').toString();
    }

    private static Fragment parse(Map<String, ?> map) {
        checkType(map, Type.FRAGMENT);
        Fragment fragment = new Fragment(getAttrOrThrow(map, Attr.LOCAL_ID, String.class));
        getAttr(map, Attr.NAME, String.class).ifPresent(fragment::name);
        return fragment;
    }


    /**
     * Types that represent a partially constructed {@link Fragment fragment}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code fragment} still needs its {@code localId} attribute set.
         */
        class NeedsLocalId {
            static final NeedsLocalId INSTANCE = new NeedsLocalId();

            private NeedsLocalId() {
            }

            public Fragment localId(String localId) {
                return new Fragment(localId);
            }
        }
    }
}
