package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.mark.DataConsumerException;
import com.atlassian.adf.model.mark.type.BodiedExtensionMark;
import com.atlassian.adf.model.mark.type.ExtensionMark;
import com.atlassian.adf.model.mark.type.InlineExtensionMark;
import com.atlassian.adf.util.Factory;

import java.util.*;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Its purpose seems to be related to giving extension nodes a way to specify that an extension will
 * want access to the data supplied by the given data sources on the page, presumably so that it can
 * render them nicely for the end user.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class DataConsumer
        extends AbstractMark
        implements BodiedExtensionMark, ExtensionMark, InlineExtensionMark {

    static final Factory<DataConsumer> FACTORY = new Factory<>(
            Type.DATA_CONSUMER,
            DataConsumer.class,
            DataConsumer::parse
    );

    // Note: We want to preserve the original iteration order
    private final Set<String> sources = new LinkedHashSet<>();

    private DataConsumer() {
    }

    public static Partial.NeedsSource dataConsumer() {
        return new Partial.NeedsSource();
    }

    public static DataConsumer dataConsumer(String source) {
        return dataConsumer(Stream.of(source));
    }

    public static DataConsumer dataConsumer(String... sources) {
        return dataConsumer(Arrays.stream(sources));
    }

    public static DataConsumer dataConsumer(Iterable<String> sources) {
        return dataConsumer(StreamSupport.stream(sources.spliterator(), false));
    }

    public static DataConsumer dataConsumer(Stream<String> sources) {
        DataConsumer dataConsumer = new DataConsumer();
        sources.forEach(dataConsumer::source);
        dataConsumer.validate();
        return dataConsumer;
    }

    @Override
    public DataConsumer copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.DATA_CONSUMER;
    }

    @Override
    public void validate() {
        super.validate();
        if (sources.isEmpty()) {
            throw new DataConsumerException.NoSources();
        }
    }

    public DataConsumer source(String source) {
        requireNonNull(source, "source");
        if (!sources.add(source)) {
            throw new DataConsumerException.DuplicateSource(source);
        }
        return this;
    }

    public Set<String> sources() {
        // Note: We don't use Set.of because we want to preserve the iteration order
        return Collections.unmodifiableSet(sources);
    }

    @Override
    public Map<String, ?> toMap() {
        List<String> sources = new ArrayList<>(this.sources);
        if (sources.isEmpty()) {
            throw new DataConsumerException.NoSources();
        }

        return mapWithType()
                .add(Key.ATTRS, map(Attr.SOURCES, sources));
    }

    @Override
    public boolean equals(Object o) {
        return this == o || (o instanceof DataConsumer && ((DataConsumer) o).sources.equals(sources));
    }

    @Override
    public int hashCode() {
        return sources.hashCode();
    }

    @Override
    public String toString() {
        return elementType() + sources;
    }

    private static DataConsumer parse(Map<String, ?> map) {
        checkType(map, Type.DATA_CONSUMER);
        List<String> sources = unsafeCast(getAttrOrThrow(map, Attr.SOURCES, List.class));
        if (sources.isEmpty()) {
            throw new DataConsumerException.NoSources();
        }

        DataConsumer dataConsumer = new DataConsumer();
        sources.forEach(dataConsumer::source);
        return dataConsumer;
    }


    /**
     * Types that represent a partially constructed {@link DataConsumer dataConsumer}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code dataConsumer} still needs at least one {@code source}.
         */
        class NeedsSource {
            NeedsSource() {
            }

            public DataConsumer source(String source) {
                return dataConsumer(source);
            }

            public DataConsumer sources(String... sources) {
                return dataConsumer(sources);
            }

            public DataConsumer sources(Iterable<String> sources) {
                return dataConsumer(sources);
            }

            public DataConsumer sources(Stream<String> sources) {
                return dataConsumer(sources);
            }
        }
    }

}
