package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.CodeTextMark;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.mark.type.TextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * The annotation mark shows the presence of an annotation or comment on a node.
 * This mark applies to {@link Text text} nodes.
 * <p>
 * Annotation marks are used in Confluence to provide links to inline comments.
 * It does not seem to be used by any other products at this time.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 *     {@link Paragraph#p(InlineContent...) p}(
 *         {@link Text#text(String) text}("hello "),
 *         {@link Text#text(String) text}("there")
 *                 .{@link Text#mark(TextMark) mark}({@link #inlineComment(String) inlineComment}("123456"))
 *     );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "paragraph",
 *    "content": [
 *      {
 *        "type": "text",
 *        "text": "hello "
 *      },
 *      {
 *        "type": "text",
 *        "text": "there",
 *        "marks": [
 *          {
 *            "type": "annotation",
 *            "attrs": {
 *              "id": "123456",
 *              "annotationType": "inlineComment"
 *            }
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <!-- Note: IntelliJ's JavaDocs viewer does not support changing the cursor style -->
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * hello <span style="background-color: #fff0c5; cursor: pointer;">there</span></span>
 * </div>
 * Note: This example output is not using AtlasKit to render the link, so while it gives a vague
 * impression of what an annotation is, it does not faithfully reproduce the actual presentation in
 * Atlassian products.
 */
@Immutable
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Annotation
        extends AbstractMark
        implements CodeTextMark, FormattedTextMark {

    static final Factory<Annotation> FACTORY = new Factory<>(Type.ANNOTATION, Annotation.class, Annotation::parse);

    private final String id;
    private final AnnotationType annotationType;

    public Annotation(String id, AnnotationType annotationType) {
        this.id = requireNonNull(id, "id");
        this.annotationType = requireNonNull(annotationType, "annotationType");
    }

    public static Annotation annotation(String id, AnnotationType annotationType) {
        return new Annotation(id, annotationType);
    }

    @Override
    public Annotation copy() {
        return this;
    }

    public static Annotation inlineComment(String id) {
        return annotation(id, AnnotationType.INLINE_COMMENT);
    }

    public String id() {
        return id;
    }

    public AnnotationType annotationType() {
        return annotationType;
    }

    @Override
    public String elementType() {
        return Type.ANNOTATION;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.ANNOTATION_TYPE, annotationType.annotationType()));
    }

    @Override
    public String toString() {
        return elementType() + "[id=" + id + ", annotationType=" + annotationType + ']';
    }

    @Override
    public boolean equals(@Nullable Object o) {
        return this == o || (o instanceof Annotation && equalTo((Annotation) o));
    }

    private boolean equalTo(Annotation other) {
        return id.equals(other.id) && annotationType == other.annotationType;
    }

    @Override
    public int hashCode() {
        return id.hashCode() * 31 + annotationType.hashCode();
    }

    private static Annotation parse(Map<String, ?> map) {
        checkType(map, Type.ANNOTATION);
        String id = getAttrOrThrow(map, Attr.ID);
        AnnotationType annotationType = AnnotationType.PARSER.parse(getAttrOrThrow(map, Attr.ANNOTATION_TYPE));
        return annotation(id, annotationType);
    }

    public enum AnnotationType {
        INLINE_COMMENT("inlineComment");

        static final EnumParser<AnnotationType> PARSER = new EnumParser<>(
                AnnotationType.class,
                AnnotationType::annotationType
        );

        private final String annotationType;

        AnnotationType(String annotationType) {
            this.annotationType = annotationType;
        }

        public String annotationType() {
            return annotationType;
        }
    }
}
