package com.atlassian.adf.model.ex.node;

import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Reports an error with a {@code tableCell} or {@code tableHeader} node.
 */
public abstract class TableCellException extends NodeException {
    private static final long serialVersionUID = 1L;

    TableCellException(String message) {
        super(message);
    }

    /**
     * Reports a problem with the {@code colwidth} attribute of a {@code tableCell} or {@code tableHeader}.
     */
    public abstract static class ColwidthException extends TableCellException {
        private static final long serialVersionUID = 1L;

        ColwidthException(String message) {
            super(message);
        }
    }

    /**
     * Reports that at least one negative value was provided in the {@code colwidth} attribute for a
     * {@code tableCell} or {@code tableHeader}.
     */
    public static class ColwidthCannotBeNegative extends ColwidthException {
        private static final long serialVersionUID = 1L;

        private final int index;
        private final int value;

        public ColwidthCannotBeNegative(int index, int value) {
            super("Value at colwidth[" + index + "] cannot be negative: " + value);
            this.index = index;
            this.value = value;
            if (index < 0) {
                throw new ArrayIndexOutOfBoundsException(index);
            }
            if (value >= 0) {
                throw new IllegalArgumentException("Exception should only be thrown for a negative value: " + value);
            }
        }

        public int index() {
            return index;
        }

        public int value() {
            return value;
        }
    }

    /**
     * Reports that all of the values {@code colwidth} attribute for a {@code tableCell} or {@code tableHeader}
     * were {@code 0}. The {@code colwidth} attribute must give at least one positive value.
     */
    public static class ColwidthMustHaveAtLeastOnePositiveValue extends ColwidthException {
        private static final long serialVersionUID = 1L;

        public ColwidthMustHaveAtLeastOnePositiveValue() {
            super("The column width attribute 'colwidth' must contain at least one positive value");
        }
    }

    /**
     * Reports that the array of values {@code colwidth} attribute for a {@code tableCell} or {@code tableHeader}
     * did not match the {@code colspan} attribute. The array must have exactly the same number of entries as
     * the value of {@code colspan}, which is taken to be {@code 1} when it is not explicitly specified.
     */
    public static class ColwidthDoesNotMatchColspan extends ColwidthException {
        private static final long serialVersionUID = 1L;

        @Nullable
        private final Integer colspan;

        private final int colwidthLength;

        public ColwidthDoesNotMatchColspan(@Nullable Integer colspan, int[] colwidth) {
            this(colspan, requireNonNull(colwidth, "colwidth").length);
        }

        private ColwidthDoesNotMatchColspan(@Nullable Integer colspan, int colwidthLength) {
            super("The 'colwidth' must contain exactly 'colspan' values, or exactly 1 value if it is not specified: "
                    + "colspan=" + colspan + "; colwidth.length=" + colwidthLength);
            this.colspan = colspan;
            this.colwidthLength = colwidthLength;
            int expectedLength = (colspan != null) ? colspan : 1;
            if (colwidthLength == expectedLength) {
                throw new IllegalArgumentException("This should only be called when colspan=" + colspan +
                        " and colwidth.length=" + colwidthLength + " do not match");
            }
        }

        /**
         * @return the value that was provided for {@code colspan}
         */
        @Nullable
        public Integer colspan() {
            return colspan;
        }

        /**
         * @return the effective value of {@code colspan}, which is in turn the expected number of items in the
         * {@code colwidth} array
         */
        public int expectedLength() {
            return (colspan != null) ? colspan : 1;
        }

        /**
         * @return the actual number of items in the {@code colwidth} array
         */
        public int colwidthLength() {
            return colwidthLength;
        }
    }

    /**
     * Reports that the {@code colspan} attribute for a {@code tableCell} or {@code tableHeader} node was
     * not positive.
     */
    public static class ColspanNotPositive extends TableCellException {
        private static final long serialVersionUID = 1L;

        private final int colspan;

        public ColspanNotPositive(int colspan) {
            super("Value for 'colspan' must be positive: " + colspan);
            this.colspan = colspan;
            if (colspan > 0) {
                throw new IllegalArgumentException("Exception should only be thrown for a non-positive value: " +
                        colspan);
            }
        }

        public int colspan() {
            return colspan;
        }
    }

    /**
     * Reports that the {@code rowspan} attribute for a {@code tableCell} or {@code tableHeader} was negative.
     */
    public static class RowspanNegative extends TableCellException {
        private static final long serialVersionUID = 1L;

        private final int rowspan;

        public RowspanNegative(int rowspan) {
            super("Value for 'rowspan' cannot be negative: " + rowspan);
            this.rowspan = rowspan;
            if (rowspan >= 0) {
                throw new IllegalArgumentException("Exception should only be thrown for a negative value: " + rowspan);
            }
        }

        public int rowspan() {
            return rowspan;
        }
    }
}
