package com.atlassian.adf.model.ex.node;

import com.atlassian.adf.model.node.LayoutSection;

import static com.atlassian.adf.model.node.LayoutSection.MAXIMUM_LAYOUT_COLUMN_COUNT;
import static com.atlassian.adf.model.node.LayoutSection.MINIMUM_LAYOUT_COLUMN_COUNT;

/**
 * Reports a problem with a {@code layoutSection} node.
 */
public class LayoutSectionException extends NodeException {
    private static final long serialVersionUID = 1L;

    LayoutSectionException(String message) {
        super(message);
    }

    /**
     * Reports a problem with the number of {@code layoutColumn} nodes contained within a {@code layoutSection}
     * node.
     */
    public static abstract class ColumnCountException extends LayoutSectionException {
        private static final long serialVersionUID = 1L;

        private final int count;

        ColumnCountException(int count, String message) {
            super(message);
            this.count = count;
            if (count >= MINIMUM_LAYOUT_COLUMN_COUNT && count <= MAXIMUM_LAYOUT_COLUMN_COUNT) {
                throw new IllegalArgumentException("This should only be called with an invalid count: " + count);
            }
        }

        public int count() {
            return count;
        }
    }

    /**
     * Indicates that a {@code layoutSection} node did not have enough {@code layoutColumn} nodes within it
     * to be valid. At least {@value LayoutSection#MINIMUM_LAYOUT_COLUMN_COUNT} of them are required.
     */
    public static class NotEnoughLayoutColumns extends ColumnCountException {
        private static final long serialVersionUID = 1L;

        public NotEnoughLayoutColumns(int count) {
            super(count, "Not enough layout columns provided: minimum=" + MINIMUM_LAYOUT_COLUMN_COUNT +
                    "; actual=" + count);
        }
    }

    /**
     * Indicates that a {@code layoutSection} node had too many {@code layoutColumn} nodes within it
     * to be valid. No more than {@value LayoutSection#MAXIMUM_LAYOUT_COLUMN_COUNT} of them are permitted.
     */
    public static class TooManyLayoutColumns extends ColumnCountException {
        private static final long serialVersionUID = 1L;

        public TooManyLayoutColumns(int count) {
            super(count, "Too many layout columns provided: maximum=" + MAXIMUM_LAYOUT_COLUMN_COUNT +
                    "; actual=" + count);
        }
    }
}
