package com.atlassian.adf.model.ex.node;

import static java.util.Objects.requireNonNull;

/**
 * Reports a problem with one of the various "card" node types: {@code blockCard}, {@code embedCard}, or
 * {@code inlineCard}.
 */
public abstract class CardException extends NodeException {
    private static final long serialVersionUID = 1L;

    CardException(String message) {
        super(message);
    }

    /**
     * Reports an attempt to set both the {@code url} and {@code data} values on a {@code blockCard} or
     * {@code inlineCard}. One of these must be set, but not both of them.
     */
    public static class UrlAndDataCannotBothBeSet extends CardException {
        private static final long serialVersionUID = 1L;

        public UrlAndDataCannotBothBeSet() {
            super("The 'url' and 'data' values cannot both be set");
        }
    }

    /**
     * Reports an attempt to use a {@code blockCard} or {@code inlineCard} without first setting
     * either its {@code url} or {@code data} value. Exactly one of these must be set.
     */
    public static class UrlOrDataMustBeSet extends CardException {
        private static final long serialVersionUID = 1L;

        public UrlOrDataMustBeSet() {
            super("Either the 'url' or 'data' value must be set");
        }
    }

    /**
     * Reports an attempt to use an invalid {@code width} value on an {@code embedCard} node.
     */
    public static class InvalidWidth extends CardException {
        private static final long serialVersionUID = 1L;

        public InvalidWidth(Number width) {
            super("The 'width' must be from 0.0 to 100.0: " + width);
            this.width = requireNonNull(width, "width");
        }

        private final Number width;

        public Number width() {
            return width;
        }
    }
}
