package com.atlassian.adf.model;

import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.node.Node;

import javax.annotation.Nullable;
import java.util.Map;

/**
 * Any object within the Atlassian Document Format (ADF) structure.
 * <p>
 * The term "element" is not defined by the JSON schema and is not used by documentation outside of this
 * library, but it has been introduced here to declare features that are common to both {@link Node nodes}
 * (which represent content) and {@link Mark marks} (which are attached to a node to alter its presentation).
 * For example, both nodes and marks must set a {@link Key#TYPE type}, so that key and the {@link #elementType()}
 * method to return its value are both declared here.
 */
public interface Element {

    /**
     * The {@code type} value that identifies this element, such as {@code "paragraph"} or {@code "strong"}.
     */
    String elementType();

    /**
     * Transforms this element to a map of {@code String} values to various basic object types suitable for direct
     * rendering as JSON content.
     *
     * @return the map representation of this ADF element
     * @throws AdfException if the element is not in a valid state. The most likely cause for this is a
     *                      content node (such as a {@code listItem}) that is not permitted to be empty.
     *                      For content nodes, the descendants are included in the generated map, so their
     *                      states are also validated.
     */
    Map<String, ?> toMap();

    /**
     * Returns a deep copy of this element, including copies of any nodes or marks that it contains.
     * The copy will not necessarily be in exactly the same state as the original in some cases.
     * For example, a {@code text} node that is used inside a {@code codeBlock} will have the
     * ability to use marks on it disabled, but a copy made of the text node using this method will
     * not similarly disallow marks unless it is also added to a content node with those same
     * restrictions.
     * <p>
     * Implementations notes:
     * <ul>
     *     <li>Implementations should narrow the return type.</li>
     *     <li>Implementations should {@code return this} if the element is immutable. The {@code @Immutable}
     *          annotation should be used on the class to offer additional confirmation of this intent.</li>
     *     <li>Implementations should {@code return parse(toMap())} if they have state.</li>
     *     <li>While there may be cases where it is worthwhile to do something more efficient than
     *          the conversion to a map and back, this is discouraged because it would add yet another
     *          fragile piece of code that breaks when new data is added to the node. The {@code parse}
     *          and {@code toMap} methods already have to be updated in these circumstances, so it
     *          makes sense to take advantage of that.</li>
     * </ul>
     *
     * @return a copy of this element, or {@code this} if the element is immutable anyway
     * @throws AdfException as for {@link #toMap()}
     */
    Element copy();

    /**
     * Constants used as property names in both
     * {@link com.atlassian.adf.model.node nodes} and
     * {@link com.atlassian.adf.model.mark marks}.
     */
    interface Key {
        /**
         * The {@code attrs} field is a map that holds additional attributes for a mark or node.
         * Some elements have required attributes while others do not.
         */
        String ATTRS = "attrs";

        /**
         * The {@code type} field specifies a string value that identifies the type of the node.
         * In the JSON schema, this corresponds to an enumerated type with only a single valid value
         * that is specific to that node class, such as {@code paragraph} for the schema's
         * {@code paragraph_node} type. Every ADF element must set this property.
         */
        String TYPE = "type";
    }

    /**
     * Constants used as names of {@link Key#ATTRS attributes} in both
     * {@link com.atlassian.adf.model.node nodes} and
     * {@link com.atlassian.adf.model.mark marks}.
     */
    interface Attr {
        /**
         * Indicates the Collection ID as used by Media Services.
         * In most places, this value is mandatory but may be left blank.
         * This unusual situation is due to the fact that it has historically been a required value,
         * but Media Services is phasing out its use.
         */
        String COLLECTION = "collection";

        /**
         * Used to indicate the color of something.
         * This is always a string value, but the exact meaning of that string varies.
         * See the color's documentation within the corresponding mark or node for more information
         * about what values are permitted for it.
         */
        String COLOR = "color";

        /**
         * Some kind of identifier.
         * The necessity and format of these identifiers varies depending on the mark or node type.
         */
        String ID = "id";

        /**
         * An indentation or nesting level, given as an integer value from {@code 1} to {@code 6}.
         */
        String LEVEL = "level";

        /**
         * Some kind of identifier.
         * The necessity and format of these identifiers varies depending on the mark or node type.
         */
        String LOCAL_ID = "localId";

        /**
         * A token provided by media services in reference to a specific use of a given piece of
         * content. It is generally optional, but required by Media Services when requesting that
         * a piece of content be deleted.
         */
        String OCCURRENCE_KEY = "occurrenceKey";

        /**
         * The size of something.
         * For example, the {@code border} mark uses this to provide the border's width in pixels.
         */
        String SIZE = "size";

        /**
         * Provides text content.
         * Note that {@code text} nodes themselves use the {@link Node.Key#TEXT "text" key} for this,
         * meaning that the value is a field directly on the node itself. The {@code text} attributes
         * are instead used in in other contexts where it provides the display or placeholder text for
         * some richer content type, such as the value displayed for a {@code status} lozenge or the
         * text shown linked to a user's profile in the case of a {@code mention}.
         */
        String TEXT = "text";

        /**
         * Provides the {@code title} text for an active page element, such as a hyperlink or an
         * expandable section.
         */
        String TITLE = "title";
    }

    /**
     * Indicates whether this element is fully supported by this library.
     * This library includes a small amount of support for preserving new, unrecognized node types
     * during a round trip. These are represented by immutable placeholder objects that return {@code false}
     * for this value.
     *
     * @return {@code true} for fully supported elements; {@code false} for immutable placeholders
     */
    boolean isSupported();

    /**
     * Verifies that the node is well-formed (including the state of any descendents that it has).
     *
     * @throws IllegalStateException if the node does not currently represent valid ADF content
     */
    void validate();

    /**
     * A convenience method used during parsing to throw a {@code MissingProperty} exception for missing
     * required values.
     *
     * @param value        the value to check
     * @param propertyName the name of the property that the value was read from
     * @param <T>          the inferred type of {@code value}
     * @return {@code value}, which has been verified as non-null
     * @throws AdfException.MissingProperty if {@code value} is {@code null}
     */
    static <T> T nonNull(@Nullable T value, String propertyName) {
        if (value == null) {
            throw new AdfException.MissingProperty(propertyName);
        }
        return value;
    }

    /**
     * A convenience method used during parsing to throw an {@code EmptyProperty} exception if the value
     * provided is an empty string, while still permitting {@code null}.
     *
     * @param value        the value to check
     * @param propertyName the name of the property that the value was read from
     * @return {@code value}, which may be {@code null}, but will not be {@code ""}
     * @throws AdfException.EmptyProperty if {@code value} is {@code ""}
     */
    @Nullable
    static String nullOrNonEmpty(@Nullable String value, String propertyName) {
        return (value != null) ? nonEmpty(value, propertyName) : null;
    }

    /**
     * A convenience method used during parsing to throw a {@code MissingProperty} exception for missing
     * required values or {@code EmptyProperty} exception if the value provided is an empty string.
     *
     * @param value        the value to check
     * @param propertyName the name of the property that the value was read from
     * @return {@code value}, which has been verified as non-null
     * @throws AdfException.MissingProperty if {@code value} is {@code null}
     * @throws AdfException.EmptyProperty   if {@code value} is {@code ""}
     */
    static String nonEmpty(String value, String propertyName) {
        nonNull(value, propertyName);
        if (value.isEmpty()) {
            throw new AdfException.EmptyProperty(propertyName);
        }
        return value;
    }
}
