package com.atlassian.adf.model;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Target;

/**
 * Expresses the library maintainer's opinion of the state of the public documentation for the
 * Atlassian Document Format with regard to the annotated node or mark.
 * <p>
 * The Atlassian Document Format (ADF) was originally created as part of the development of
 * <a href="https://en.wikipedia.org/wiki/Stride_%28software%29">Atlassian Stride</a>, the
 * intended successor to <a href="https://en.wikipedia.org/wiki/HipChat">HipChat</a> that was
 * later abandoned in favor of <a href="https://www.atlassian.com/partnerships/slack">a strategic
 * partnership</a> with <a href="https://en.wikipedia.org/wiki/Slack_%28software%29">Slack</a>.
 * <p>
 * When Jira began the process of adopting ADF, Stride's existing documentation for it was copied
 * into the
 * <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/">Jira Cloud
 * REST API</a> documentation, where it can still be found. The plan was for all products to
 * keep their own copy of this documentation, tailored to the feature set in each product. For
 * example, {@code layoutColumn} nodes are a feature of how Confluence uses ADF and are not
 * available in Jira, so Jira does not include them in its copy of the documentation.
 * <p>
 * Unfortunately, Stride was divested in the middle of this process, and the copy of the
 * documentation that Jira made in mid-2018 has not been updated since. Worse, none of the
 * other products ever incorporated the Stride copy of the documentation into their own REST
 * APIs, and they all point at either Stride's original URL (which is no longer available) or
 * at Jira's unmaintained copy.
 * <p>
 * The good news is that the original source for the documentation still exists and (with a few oversights)
 * has been maintained. We just aren't publishing it anywhere, yet. Hopefully we will fix this soon, but
 * in the meantime, this annotation is intended to help call out where the public documentation is known to
 * fall short.
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/">Jira
 * Cloud REST API &mdash; Atlassian Document Format &mdash; Document structure</a>
 */
@Documented
@Target(ElementType.TYPE)
public @interface Documentation {
    /**
     * A simple status indicator for the health of the public documentation regarding this item.
     */
    State state();

    /**
     * The date on which the documentation was last reviewed.
     */
    String date();

    /**
     * Additional information about the nature of the problem with the existing public documentation.
     */
    String comment() default "";

    /**
     * Expresses that library's author on the state of the
     * <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/structure/">Jira ADF
     * Documentation</a> for this item (whether it is a mark, a node, or a property of either) at the
     * time of its more recent review on {@link #date()}.
     */
    enum State {
        /**
         * The contents were complete and up-to-date as of the review date.
         */
        REVIEWED,

        /**
         * There is no existing documentation of this feature.
         */
        UNDOCUMENTED,

        /**
         * The existing documentation is correct, but some of the features are not included in the documentation
         * or their values are not fully explained.
         */
        INCOMPLETE,

        /**
         * The existing documentation contains information that is incorrect or misleading.
         */
        WRONG
    }
}