/*
 * Decompiled with CFR 0.152.
 */
package com.arxanfintech.common.crypto.core;

import com.arxanfintech.common.crypto.core.Constants;
import com.arxanfintech.common.crypto.core.jce.ECKeyAgreement;
import com.arxanfintech.common.crypto.core.jce.ECKeyFactory;
import com.arxanfintech.common.crypto.core.jce.ECKeyPairGenerator;
import com.arxanfintech.common.crypto.core.jce.ECSignatureFactory;
import com.arxanfintech.common.crypto.core.jce.SpongyCastleProvider;
import com.arxanfintech.common.util.BIUtil;
import com.arxanfintech.common.util.ByteUtil;
import com.arxanfintech.common.util.HashUtil;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECPoint;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.Arrays;
import javax.crypto.KeyAgreement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.asn1.ASN1InputStream;
import org.spongycastle.asn1.ASN1Integer;
import org.spongycastle.asn1.DLSequence;
import org.spongycastle.asn1.sec.SECNamedCurves;
import org.spongycastle.asn1.x9.X9ECParameters;
import org.spongycastle.asn1.x9.X9IntegerConverter;
import org.spongycastle.crypto.BlockCipher;
import org.spongycastle.crypto.CipherParameters;
import org.spongycastle.crypto.Digest;
import org.spongycastle.crypto.agreement.ECDHBasicAgreement;
import org.spongycastle.crypto.digests.SHA256Digest;
import org.spongycastle.crypto.engines.AESFastEngine;
import org.spongycastle.crypto.modes.SICBlockCipher;
import org.spongycastle.crypto.params.ECDomainParameters;
import org.spongycastle.crypto.params.ECPrivateKeyParameters;
import org.spongycastle.crypto.params.ECPublicKeyParameters;
import org.spongycastle.crypto.params.KeyParameter;
import org.spongycastle.crypto.params.ParametersWithIV;
import org.spongycastle.crypto.signers.DSAKCalculator;
import org.spongycastle.crypto.signers.ECDSASigner;
import org.spongycastle.crypto.signers.HMacDSAKCalculator;
import org.spongycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.spongycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.spongycastle.jce.spec.ECParameterSpec;
import org.spongycastle.jce.spec.ECPrivateKeySpec;
import org.spongycastle.jce.spec.ECPublicKeySpec;
import org.spongycastle.math.ec.ECAlgorithms;
import org.spongycastle.math.ec.ECCurve;
import org.spongycastle.math.ec.ECPoint;
import org.spongycastle.util.BigIntegers;
import org.spongycastle.util.encoders.Base64;
import org.spongycastle.util.encoders.Hex;

public class ECKey
implements Serializable {
    private static final Logger logger = LoggerFactory.getLogger(ECKey.class);
    public static final ECDomainParameters CURVE;
    public static final ECParameterSpec CURVE_SPEC;
    public static final BigInteger HALF_CURVE_ORDER;
    private static final SecureRandom secureRandom;
    private static final long serialVersionUID = -728224901792295832L;
    private final PrivateKey privKey;
    protected final org.spongycastle.math.ec.ECPoint pub;
    private final Provider provider;
    private transient byte[] pubKeyHash;
    private transient byte[] nodeId;

    public ECKey() {
        this(secureRandom);
    }

    public static org.spongycastle.math.ec.ECPoint extractPublicKey(ECPublicKey ecPublicKey) {
        ECPoint publicPointW = ecPublicKey.getW();
        BigInteger xCoord = publicPointW.getAffineX();
        BigInteger yCoord = publicPointW.getAffineY();
        return CURVE.getCurve().createPoint(xCoord, yCoord);
    }

    public ECKey(Provider provider, SecureRandom secureRandom) {
        this.provider = provider;
        KeyPairGenerator keyPairGen = ECKeyPairGenerator.getInstance(provider, secureRandom);
        KeyPair keyPair = keyPairGen.generateKeyPair();
        this.privKey = keyPair.getPrivate();
        PublicKey pubKey = keyPair.getPublic();
        if (pubKey instanceof BCECPublicKey) {
            this.pub = ((BCECPublicKey)pubKey).getQ();
        } else if (pubKey instanceof ECPublicKey) {
            this.pub = ECKey.extractPublicKey((ECPublicKey)pubKey);
        } else {
            throw new AssertionError((Object)("Expected Provider " + provider.getName() + " to produce a subtype of ECPublicKey, found " + pubKey.getClass()));
        }
    }

    public ECKey(SecureRandom secureRandom) {
        this(SpongyCastleProvider.getInstance(), secureRandom);
    }

    private static boolean isECPrivateKey(PrivateKey privKey) {
        return privKey instanceof ECPrivateKey || privKey.getAlgorithm().equals("EC");
    }

    public ECKey(Provider provider, PrivateKey privKey, org.spongycastle.math.ec.ECPoint pub) {
        this.provider = provider;
        if (privKey != null && !ECKey.isECPrivateKey(privKey)) {
            throw new IllegalArgumentException("Expected EC private key, given a private key object with class " + privKey.getClass().toString() + " and algorithm " + privKey.getAlgorithm());
        }
        this.privKey = privKey;
        if (pub == null) {
            throw new IllegalArgumentException("Public key may not be null");
        }
        this.pub = pub;
    }

    private static PrivateKey privateKeyFromBigInteger(BigInteger priv) {
        if (priv == null) {
            return null;
        }
        try {
            return ECKeyFactory.getInstance(SpongyCastleProvider.getInstance()).generatePrivate((KeySpec)new ECPrivateKeySpec(priv, CURVE_SPEC));
        }
        catch (InvalidKeySpecException ex) {
            throw new AssertionError((Object)"Assumed correct key spec statically");
        }
    }

    public ECKey(BigInteger priv, org.spongycastle.math.ec.ECPoint pub) {
        this(SpongyCastleProvider.getInstance(), ECKey.privateKeyFromBigInteger(priv), pub);
    }

    public static org.spongycastle.math.ec.ECPoint compressPoint(org.spongycastle.math.ec.ECPoint uncompressed) {
        return CURVE.getCurve().decodePoint(uncompressed.getEncoded(true));
    }

    public static org.spongycastle.math.ec.ECPoint decompressPoint(org.spongycastle.math.ec.ECPoint compressed) {
        return CURVE.getCurve().decodePoint(compressed.getEncoded(false));
    }

    public static ECKey fromPrivate(BigInteger privKey) {
        return new ECKey(privKey, CURVE.getG().multiply(privKey));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes) {
        return ECKey.fromPrivate(new BigInteger(1, privKeyBytes));
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(BigInteger priv, org.spongycastle.math.ec.ECPoint pub) {
        return new ECKey(priv, pub);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(byte[] priv, byte[] pub) {
        ECKey.check(priv != null, "Private key must not be null");
        ECKey.check(pub != null, "Public key must not be null");
        return new ECKey(new BigInteger(1, priv), CURVE.getCurve().decodePoint(pub));
    }

    public static ECKey fromPublicOnly(org.spongycastle.math.ec.ECPoint pub) {
        return new ECKey(null, pub);
    }

    public static ECKey fromPublicOnly(byte[] pub) {
        return new ECKey(null, CURVE.getCurve().decodePoint(pub));
    }

    public ECKey decompress() {
        if (!this.pub.isCompressed()) {
            return this;
        }
        return new ECKey(this.provider, this.privKey, ECKey.decompressPoint(this.pub));
    }

    public ECKey compress() {
        if (this.pub.isCompressed()) {
            return this;
        }
        return new ECKey(this.provider, this.privKey, ECKey.compressPoint(this.pub));
    }

    public boolean isPubKeyOnly() {
        return this.privKey == null;
    }

    public boolean hasPrivKey() {
        return this.privKey != null;
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey, boolean compressed) {
        org.spongycastle.math.ec.ECPoint point = CURVE.getG().multiply(privKey);
        return point.getEncoded(compressed);
    }

    public static byte[] computeAddress(byte[] pubBytes) {
        return HashUtil.sha3omit12(Arrays.copyOfRange(pubBytes, 1, pubBytes.length));
    }

    public static byte[] computeAddress(org.spongycastle.math.ec.ECPoint pubPoint) {
        return ECKey.computeAddress(pubPoint.getEncoded(false));
    }

    public byte[] getAddress() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = ECKey.computeAddress(this.pub);
        }
        return this.pubKeyHash;
    }

    public static byte[] pubBytesWithoutFormat(org.spongycastle.math.ec.ECPoint pubPoint) {
        byte[] pubBytes = pubPoint.getEncoded(false);
        return Arrays.copyOfRange(pubBytes, 1, pubBytes.length);
    }

    public byte[] getNodeId() {
        if (this.nodeId == null) {
            this.nodeId = ECKey.pubBytesWithoutFormat(this.pub);
        }
        return this.nodeId;
    }

    public static ECKey fromNodeId(byte[] nodeId) {
        ECKey.check(nodeId.length == 64, "Expected a 64 byte node id");
        byte[] pubBytes = new byte[65];
        System.arraycopy(nodeId, 0, pubBytes, 1, nodeId.length);
        pubBytes[0] = 4;
        return ECKey.fromPublicOnly(pubBytes);
    }

    public byte[] getPubKey() {
        return this.pub.getEncoded(false);
    }

    public org.spongycastle.math.ec.ECPoint getPubKeyPoint() {
        return this.pub;
    }

    public BigInteger getPrivKey() {
        if (this.privKey == null) {
            throw new MissingPrivateKeyException();
        }
        if (this.privKey instanceof BCECPrivateKey) {
            return ((BCECPrivateKey)this.privKey).getD();
        }
        throw new MissingPrivateKeyException();
    }

    public boolean isCompressed() {
        return this.pub.isCompressed();
    }

    public String toString() {
        StringBuilder b = new StringBuilder();
        b.append("pub:").append(Hex.toHexString((byte[])this.pub.getEncoded(false)));
        return b.toString();
    }

    public String toStringWithPrivate() {
        StringBuilder b = new StringBuilder();
        b.append(this.toString());
        if (this.privKey != null && this.privKey instanceof BCECPrivateKey) {
            b.append(" priv:").append(Hex.toHexString((byte[])((BCECPrivateKey)this.privKey).getD().toByteArray()));
        }
        return b.toString();
    }

    public ECDSASignature doSign(byte[] input) {
        if (input.length != 32) {
            throw new IllegalArgumentException("Expected 32 byte input to ECDSA signature, not " + input.length);
        }
        if (this.privKey == null) {
            throw new MissingPrivateKeyException();
        }
        if (this.privKey instanceof BCECPrivateKey) {
            ECDSASigner signer = new ECDSASigner((DSAKCalculator)new HMacDSAKCalculator((Digest)new SHA256Digest()));
            ECPrivateKeyParameters privKeyParams = new ECPrivateKeyParameters(((BCECPrivateKey)this.privKey).getD(), CURVE);
            signer.init(true, (CipherParameters)privKeyParams);
            BigInteger[] components = signer.generateSignature(input);
            return new ECDSASignature(components[0], components[1]).toCanonicalised();
        }
        try {
            Signature ecSig = ECSignatureFactory.getRawInstance(this.provider);
            ecSig.initSign(this.privKey);
            ecSig.update(input);
            byte[] derSignature = ecSig.sign();
            return ECDSASignature.decodeFromDER(derSignature).toCanonicalised();
        }
        catch (InvalidKeyException | SignatureException ex) {
            throw new RuntimeException("ECKey signing error", ex);
        }
    }

    public ECDSASignature sign(byte[] messageHash) {
        ECDSASignature sig = this.doSign(messageHash);
        int recId = -1;
        byte[] thisKey = this.pub.getEncoded(false);
        for (int i = 0; i < 4; ++i) {
            byte[] k = ECKey.recoverPubBytesFromSignature(i, sig, messageHash);
            if (k == null || !Arrays.equals(k, thisKey)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        sig.v = (byte)(recId + 27);
        return sig;
    }

    public static byte[] signatureToKeyBytes(byte[] messageHash, String signatureBase64) throws SignatureException {
        byte[] signatureEncoded;
        try {
            signatureEncoded = Base64.decode((String)signatureBase64);
        }
        catch (RuntimeException e) {
            throw new SignatureException("Could not decode base64", e);
        }
        if (signatureEncoded.length < 65) {
            throw new SignatureException("Signature truncated, expected 65 bytes and got " + signatureEncoded.length);
        }
        return ECKey.signatureToKeyBytes(messageHash, ECDSASignature.fromComponents(Arrays.copyOfRange(signatureEncoded, 1, 33), Arrays.copyOfRange(signatureEncoded, 33, 65), (byte)(signatureEncoded[0] & 0xFF)));
    }

    public static byte[] signatureToKeyBytes(byte[] messageHash, ECDSASignature sig) throws SignatureException {
        int recId;
        byte[] key;
        ECKey.check(messageHash.length == 32, "messageHash argument has length " + messageHash.length);
        int header = sig.v;
        if (header < 27 || header > 34) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        if (header >= 31) {
            header -= 4;
        }
        if ((key = ECKey.recoverPubBytesFromSignature(recId = header - 27, sig, messageHash)) == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    public static byte[] signatureToAddress(byte[] messageHash, String signatureBase64) throws SignatureException {
        return ECKey.computeAddress(ECKey.signatureToKeyBytes(messageHash, signatureBase64));
    }

    public static byte[] signatureToAddress(byte[] messageHash, ECDSASignature sig) throws SignatureException {
        return ECKey.computeAddress(ECKey.signatureToKeyBytes(messageHash, sig));
    }

    public static ECKey signatureToKey(byte[] messageHash, String signatureBase64) throws SignatureException {
        byte[] keyBytes = ECKey.signatureToKeyBytes(messageHash, signatureBase64);
        return ECKey.fromPublicOnly(keyBytes);
    }

    public static ECKey signatureToKey(byte[] messageHash, ECDSASignature sig) throws SignatureException {
        byte[] keyBytes = ECKey.signatureToKeyBytes(messageHash, sig);
        return ECKey.fromPublicOnly(keyBytes);
    }

    public BigInteger keyAgreement(org.spongycastle.math.ec.ECPoint otherParty) {
        if (this.privKey == null) {
            throw new MissingPrivateKeyException();
        }
        if (this.privKey instanceof BCECPrivateKey) {
            ECDHBasicAgreement agreement = new ECDHBasicAgreement();
            agreement.init((CipherParameters)new ECPrivateKeyParameters(((BCECPrivateKey)this.privKey).getD(), CURVE));
            return agreement.calculateAgreement((CipherParameters)new ECPublicKeyParameters(otherParty, CURVE));
        }
        try {
            KeyAgreement agreement = ECKeyAgreement.getInstance(this.provider);
            agreement.init(this.privKey);
            agreement.doPhase(ECKeyFactory.getInstance(this.provider).generatePublic((KeySpec)new ECPublicKeySpec(otherParty, CURVE_SPEC)), true);
            return new BigInteger(1, agreement.generateSecret());
        }
        catch (IllegalStateException | InvalidKeyException | InvalidKeySpecException ex) {
            throw new RuntimeException("ECDH key agreement failure", ex);
        }
    }

    public byte[] decryptAES(byte[] cipher) {
        if (this.privKey == null) {
            throw new MissingPrivateKeyException();
        }
        if (!(this.privKey instanceof BCECPrivateKey)) {
            throw new UnsupportedOperationException("Cannot use the private key as an AES key");
        }
        AESFastEngine engine = new AESFastEngine();
        SICBlockCipher ctrEngine = new SICBlockCipher((BlockCipher)engine);
        KeyParameter key = new KeyParameter(BigIntegers.asUnsignedByteArray((BigInteger)((BCECPrivateKey)this.privKey).getD()));
        ParametersWithIV params = new ParametersWithIV((CipherParameters)key, new byte[16]);
        ctrEngine.init(false, (CipherParameters)params);
        int i = 0;
        byte[] out = new byte[cipher.length];
        while (i < cipher.length) {
            ctrEngine.processBlock(cipher, i, out, i);
            if (cipher.length - (i += engine.getBlockSize()) >= engine.getBlockSize()) continue;
        }
        if (cipher.length - i > 0) {
            byte[] tmpBlock = new byte[16];
            System.arraycopy(cipher, i, tmpBlock, 0, cipher.length - i);
            ctrEngine.processBlock(tmpBlock, 0, tmpBlock, 0);
            System.arraycopy(tmpBlock, 0, out, i, cipher.length - i);
        }
        return out;
    }

    public static boolean verify(byte[] data, ECDSASignature signature, byte[] pub) {
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(CURVE.getCurve().decodePoint(pub), CURVE);
        signer.init(false, (CipherParameters)params);
        try {
            return signer.verifySignature(data, signature.r, signature.s);
        }
        catch (NullPointerException npe) {
            logger.error("Caught NPE inside bouncy castle", (Throwable)npe);
            return false;
        }
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) {
        return ECKey.verify(data, ECDSASignature.decodeFromDER(signature), pub);
    }

    public boolean verify(byte[] data, byte[] signature) {
        return ECKey.verify(data, signature, this.getPubKey());
    }

    public boolean verify(byte[] sigHash, ECDSASignature signature) {
        return ECKey.verify(sigHash, signature, this.getPubKey());
    }

    public boolean isPubKeyCanonical() {
        return ECKey.isPubKeyCanonical(this.pub.getEncoded(false));
    }

    public static boolean isPubKeyCanonical(byte[] pubkey) {
        if (pubkey[0] == 4) {
            if (pubkey.length != 65) {
                return false;
            }
        } else if (pubkey[0] == 2 || pubkey[0] == 3) {
            if (pubkey.length != 33) {
                return false;
            }
        } else {
            return false;
        }
        return true;
    }

    public static byte[] recoverPubBytesFromSignature(int recId, ECDSASignature sig, byte[] messageHash) {
        ECKey.check(recId >= 0, "recId must be positive");
        ECKey.check(sig.r.signum() >= 0, "r must be positive");
        ECKey.check(sig.s.signum() >= 0, "s must be positive");
        ECKey.check(messageHash != null, "messageHash must not be null");
        BigInteger n = CURVE.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        ECCurve.Fp curve = (ECCurve.Fp)CURVE.getCurve();
        BigInteger prime = curve.getQ();
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        org.spongycastle.math.ec.ECPoint R = ECKey.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = new BigInteger(1, messageHash);
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint.Fp q = (ECPoint.Fp)ECAlgorithms.sumOfTwoMultiplies((org.spongycastle.math.ec.ECPoint)CURVE.getG(), (BigInteger)eInvrInv, (org.spongycastle.math.ec.ECPoint)R, (BigInteger)srInv);
        return q.getEncoded(false);
    }

    public static byte[] recoverAddressFromSignature(int recId, ECDSASignature sig, byte[] messageHash) {
        byte[] pubBytes = ECKey.recoverPubBytesFromSignature(recId, sig, messageHash);
        if (pubBytes == null) {
            return null;
        }
        return ECKey.computeAddress(pubBytes);
    }

    public static ECKey recoverFromSignature(int recId, ECDSASignature sig, byte[] messageHash) {
        byte[] pubBytes = ECKey.recoverPubBytesFromSignature(recId, sig, messageHash);
        if (pubBytes == null) {
            return null;
        }
        return ECKey.fromPublicOnly(pubBytes);
    }

    private static org.spongycastle.math.ec.ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return CURVE.getCurve().decodePoint(compEnc);
    }

    public byte[] getPrivKeyBytes() {
        if (this.privKey == null) {
            return null;
        }
        if (this.privKey instanceof BCECPrivateKey) {
            return ByteUtil.bigIntegerToBytes(((BCECPrivateKey)this.privKey).getD(), 32);
        }
        return null;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof ECKey)) {
            return false;
        }
        ECKey ecKey = (ECKey)o;
        if (this.privKey != null && !this.privKey.equals(ecKey.privKey)) {
            return false;
        }
        return this.pub == null || this.pub.equals(ecKey.pub);
    }

    public int hashCode() {
        return Arrays.hashCode(this.getPubKey());
    }

    private static void check(boolean test, String message) {
        if (!test) {
            throw new IllegalArgumentException(message);
        }
    }

    static {
        X9ECParameters params = SECNamedCurves.getByName((String)"secp256r1");
        CURVE = new ECDomainParameters(params.getCurve(), params.getG(), params.getN(), params.getH());
        CURVE_SPEC = new ECParameterSpec(params.getCurve(), params.getG(), params.getN(), params.getH());
        HALF_CURVE_ORDER = params.getN().shiftRight(1);
        secureRandom = new SecureRandom();
    }

    public static class MissingPrivateKeyException
    extends RuntimeException {
    }

    public static class ECDSASignature {
        public final BigInteger r;
        public final BigInteger s;
        public byte v;

        public ECDSASignature(BigInteger r, BigInteger s) {
            this.r = r;
            this.s = s;
        }

        private static ECDSASignature fromComponents(byte[] r, byte[] s) {
            return new ECDSASignature(new BigInteger(1, r), new BigInteger(1, s));
        }

        public static ECDSASignature fromComponents(byte[] r, byte[] s, byte v) {
            ECDSASignature signature = ECDSASignature.fromComponents(r, s);
            signature.v = v;
            return signature;
        }

        public boolean validateComponents() {
            return ECDSASignature.validateComponents(this.r, this.s, this.v);
        }

        public static boolean validateComponents(BigInteger r, BigInteger s, byte v) {
            if (v != 27 && v != 28) {
                return false;
            }
            if (BIUtil.isLessThan(r, BigInteger.ONE)) {
                return false;
            }
            if (BIUtil.isLessThan(s, BigInteger.ONE)) {
                return false;
            }
            if (!BIUtil.isLessThan(r, Constants.getSECP256K1N())) {
                return false;
            }
            return BIUtil.isLessThan(s, Constants.getSECP256K1N());
        }

        public static ECDSASignature decodeFromDER(byte[] bytes) {
            ASN1InputStream decoder = null;
            try {
                ASN1Integer s;
                ASN1Integer r;
                decoder = new ASN1InputStream(bytes);
                DLSequence seq = (DLSequence)decoder.readObject();
                if (seq == null) {
                    throw new RuntimeException("Reached past end of ASN.1 stream.");
                }
                try {
                    r = (ASN1Integer)seq.getObjectAt(0);
                    s = (ASN1Integer)seq.getObjectAt(1);
                }
                catch (ClassCastException e) {
                    throw new IllegalArgumentException(e);
                }
                ECDSASignature eCDSASignature = new ECDSASignature(r.getPositiveValue(), s.getPositiveValue());
                return eCDSASignature;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            finally {
                if (decoder != null) {
                    try {
                        decoder.close();
                    }
                    catch (IOException iOException) {}
                }
            }
        }

        public ECDSASignature toCanonicalised() {
            if (this.s.compareTo(HALF_CURVE_ORDER) > 0) {
                return new ECDSASignature(this.r, CURVE.getN().subtract(this.s));
            }
            return this;
        }

        public String toBase64() {
            byte[] sigData = new byte[65];
            sigData[0] = this.v;
            System.arraycopy(ByteUtil.bigIntegerToBytes(this.r, 32), 0, sigData, 1, 32);
            System.arraycopy(ByteUtil.bigIntegerToBytes(this.s, 32), 0, sigData, 33, 32);
            return new String(Base64.encode((byte[])sigData), Charset.forName("UTF-8"));
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ECDSASignature signature = (ECDSASignature)o;
            if (!this.r.equals(signature.r)) {
                return false;
            }
            return this.s.equals(signature.s);
        }

        public int hashCode() {
            int result = this.r.hashCode();
            result = 31 * result + this.s.hashCode();
            return result;
        }
    }
}

