package com.arthurivanets.adapster.databinding

import androidx.databinding.ObservableList
import com.arthurivanets.adapster.databinding.utils.forEachInRange
import com.arthurivanets.adapster.databinding.utils.ifCanNotifyDataSetListeners
import com.arthurivanets.adapster.model.BaseItem
import com.arthurivanets.adapster.recyclerview.TrackableRecyclerViewAdapter

/**
 *
 */
open class ObservableTrackableRecyclerViewAdapterDataSetChangeHandler<KT, IT : BaseItem<*, *, *>> : ObservableList.OnListChangedCallback<ObservableList<IT>>() {


    private var adapter : TrackableRecyclerViewAdapter<KT, IT, *>? = null


    fun attachAdapter(adapter : TrackableRecyclerViewAdapter<KT, IT, *>) {
        this.adapter = adapter
    }


    fun detachAdapter() {
        this.adapter = null
    }


    override fun onChanged(sender : ObservableList<IT>?) {
        adapter?.ifCanNotifyDataSetListeners { sender?.let(::notifyDataSetReplaced) }
        adapter?.notifyDataSetChanged()
    }


    override fun onItemRangeInserted(sender : ObservableList<IT>?, positionStart : Int, itemCount : Int) {
        adapter?.ifCanNotifyDataSetListeners {
            sender?.forEachInRange(
                startIndex = positionStart,
                endIndex = (positionStart + itemCount - 1),
                block = ::notifyDataSetItemAdded
            )
        }
        adapter?.notifyItemRangeInserted(positionStart, itemCount)
    }


    override fun onItemRangeRemoved(sender : ObservableList<IT>?, positionStart : Int, itemCount : Int) {
        adapter?.ifCanNotifyDataSetListeners {
            val dataSetItemCount = (sender?.size ?: 0)
            notifyDataSetSizeChanged((dataSetItemCount + itemCount), dataSetItemCount)
        }
        adapter?.notifyItemRangeRemoved(positionStart, itemCount)
    }


    override fun onItemRangeMoved(sender : ObservableList<IT>?, fromPosition : Int, toPosition : Int, itemCount : Int) {
        onItemRangeRemoved(sender, fromPosition, itemCount)
        onItemRangeInserted(sender, toPosition, itemCount)
    }


    override fun onItemRangeChanged(sender : ObservableList<IT>?, positionStart : Int, itemCount : Int) {
        adapter?.ifCanNotifyDataSetListeners {
            sender?.forEachInRange(
                startIndex = positionStart,
                endIndex = (positionStart + itemCount - 1),
                block = ::notifyDataSetItemUpdated
            )
        }
        adapter?.notifyItemRangeChanged(positionStart, itemCount)
    }


}