package com.artear.ui.views

import android.content.Context
import android.graphics.Canvas
import android.graphics.Paint
import android.util.AttributeSet
import androidx.appcompat.widget.AppCompatTextView
import com.artear.ui.R


class UnderlinedTextView
@JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) :
        AppCompatTextView(context, attrs, defStyleAttr) {

    private lateinit var mPaint: Paint
    private var mStrokeWidth: Float = 0.toFloat()
    private var mEnabled: Boolean = true

    var underLineColor: Int
        get() = mPaint.color
        set(mColor) {
            mPaint.color = mColor
            invalidate()
        }

    var underlineWidth: Float
        get() = mStrokeWidth
        set(mStrokeWidth) {
            this.mStrokeWidth = mStrokeWidth
            invalidate()
        }

    var underlineEnabled: Boolean
        get() = mEnabled
        set(mEnabled) {
            this.mEnabled = mEnabled
            invalidate()
        }

    init {
        init(context, attrs, defStyleAttr)
    }

    private fun init(context: Context, attributeSet: AttributeSet?, defStyle: Int) {
        if (!isInEditMode) {
            val density = context.resources.displayMetrics.density

            if (attributeSet == null || !mEnabled) {
                return
            }

            val typedArray = context.obtainStyledAttributes(attributeSet, R.styleable.UnderlinedTextView, defStyle, 0)
            val underlineColor = typedArray.getColor(R.styleable.UnderlinedTextView_underlineColor, -0x10000)
            mStrokeWidth = typedArray.getDimension(R.styleable.UnderlinedTextView_underlineWidth, density * 2)
            typedArray.recycle()

            mPaint = Paint()
            mPaint.style = Paint.Style.STROKE
            mPaint.color = underlineColor
            mPaint.strokeWidth = mStrokeWidth
        }
    }

    override fun onDraw(canvas: Canvas) {
        if (!isInEditMode && mEnabled) {
            val count = if (lineCount > maxLines) maxLines else lineCount
            var xStart: Float
            var xStop: Float

            val density = context.resources.displayMetrics.density

            for (i in 0 until count) {
                val baseline = getLineBounds(i, null)

                val diff = (layout.width - layout.getLineWidth(i)) / 2

                xStart = diff + paddingLeft
                xStop = layout.width - diff + paddingLeft

                val valueY = baseline + mStrokeWidth + (5 * density)
                canvas.drawLine(xStart, valueY, xStop, valueY, mPaint)

            }
            super.onDraw(canvas)
        }
    }
}