package com.artear.tools.preferences

import android.content.SharedPreferences
import java.util.*

class PrefsHelper private constructor(private val mPreferences: SharedPreferences) {

    var internalPreferences: InternalPreferences = InternalPreferences(mPreferences)

    private val preferencesMap: MutableMap<String, WrapperPreference<*>> = HashMap()

    fun getBooleanPreference(key: String): BooleanPreference {
        valid(key)
        return preferencesMap[key] as BooleanPreference
    }

    fun getStringPreference(key: String): StringPreference {
        valid(key)
        return preferencesMap[key] as StringPreference
    }

    fun getIntPreference(key: String): IntPreference {
        valid(key)
        return preferencesMap[key] as IntPreference
    }

    fun getLongPreference(key: String): LongPreference {
        valid(key)
        return preferencesMap[key] as LongPreference
    }

    internal fun initPreferences() {
        val keys = preferencesMap.keys
        for (key in keys) {
            if (!preferencesMap[key]!!.isSet) preferencesMap[key]!!.init()
        }
    }

    private fun valid(key: String) {
        if (!preferencesMap.containsKey(key)) {
            throw IllegalStateException("PrefsHelper - Can not find preference with key = $key")
        }
    }

    private fun put(key: String, wrapperPreference: WrapperPreference<*>) {
        preferencesMap[key] = wrapperPreference
    }

    class Builder(preferences: SharedPreferences) {

        private val prefsHelper: PrefsHelper = PrefsHelper(preferences)

        fun addBooleanPreference(key: String, value: Boolean): Builder {
            prefsHelper.put(key, BooleanPreference(prefsHelper.mPreferences, key, value))
            return this
        }

        fun addStringPreference(key: String, value: String): Builder {
            prefsHelper.put(key, StringPreference(prefsHelper.mPreferences, key, value))
            return this
        }

        fun addIntegerPreference(key: String, value: Int): Builder {
            prefsHelper.put(key, IntPreference(prefsHelper.mPreferences, key, value))
            return this
        }

        fun addLongPreference(key: String, value: Long) : PrefsHelper.Builder{
            prefsHelper.put(key, LongPreference(prefsHelper.mPreferences, key, value))
            return this
        }

        private fun valid() {
            if (prefsHelper.preferencesMap.isEmpty()) {
                throw IllegalStateException("Can not build an PrefsHelper without preferences")
            }
        }

        fun build(): PrefsHelper {
            valid()
            return prefsHelper
        }

    }

}
