/*
 * Copyright 2018 Artear S.A.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.artear.tools.network

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkInfo
import android.telephony.TelephonyManager
import android.telephony.TelephonyManager.*
import com.artear.tools.AndroidUtils
import java.util.*

object ConnectionUtil {

    /**
     * Check if there is any connectivity
     *
     * @param context
     * @return is Device Connected
     */
    fun isConnected(context: Context): Boolean {

        try {
            val cm: ConnectivityManager? = context
                    .getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

            if (null != cm) {
                val activeNetwork = cm.activeNetworkInfo

                return activeNetwork != null && activeNetwork.isConnectedOrConnecting
            }

        } catch (ignored: Exception) {

        }

        return false
    }

    //WiFi Check
    //Mobile check
    //from android TelephonyManager.getNetworkClass() method
    @Suppress("deprecation")
    fun connectionType(context: Context): ConnectionType {

        val connectivityManager: ConnectivityManager = context.getSystemService(
                Context.CONNECTIVITY_SERVICE) as ConnectivityManager?
                ?: return ConnectionType.UNKNOWN

        if (AndroidUtils.isLollipop()) {
            val networksArray = connectivityManager.allNetworks
            val networks = Arrays.asList(*networksArray)
            for (network in networks) {
                val networkInfo = connectivityManager.getNetworkInfo(network)
                if (networkInfo.isWifi()) return ConnectionType.WIFI
                return baseConnectionType(networkInfo, context)
            }
        } else {
            var networkInfo = connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI)
            if (networkInfo.isWifi()) return ConnectionType.WIFI
            networkInfo = connectivityManager.getNetworkInfo(ConnectivityManager.TYPE_MOBILE)
            return baseConnectionType(networkInfo, context)
        }

        return ConnectionType.UNKNOWN
    }

    private fun baseConnectionType(networkInfo: NetworkInfo, context: Context): ConnectionType {

        if (networkInfo.isMobile()) {

            val telephonyManager = context.getSystemService(
                    Context.TELEPHONY_SERVICE) as TelephonyManager

            val networkType = telephonyManager.networkType
            when (networkType) {
                NETWORK_TYPE_GPRS,
                NETWORK_TYPE_EDGE,
                NETWORK_TYPE_CDMA,
                NETWORK_TYPE_1xRTT,
                NETWORK_TYPE_IDEN -> {
                    return ConnectionType._2G
                }

                NETWORK_TYPE_UMTS,
                NETWORK_TYPE_EVDO_0,
                NETWORK_TYPE_EVDO_A,
                NETWORK_TYPE_HSDPA,
                NETWORK_TYPE_HSUPA,
                NETWORK_TYPE_HSPA,
                NETWORK_TYPE_EVDO_B,
                NETWORK_TYPE_EHRPD,
                NETWORK_TYPE_HSPAP -> {
                    return ConnectionType._3G
                }

                NETWORK_TYPE_LTE -> return ConnectionType._4G
            }
        }

        return ConnectionType.UNKNOWN
    }

    enum class ConnectionType {
        UNKNOWN, WIFI, _2G, _3G, _4G
    }
}

private fun NetworkInfo.isMobile(): Boolean {
    return type == ConnectivityManager.TYPE_MOBILE && isConnectedOrConnecting
}

fun NetworkInfo.isWifi(): Boolean {
    return type == ConnectivityManager.TYPE_WIFI && isConnectedOrConnecting
}