/**
 * Copyright (C) 2017-2018  Ardika Rommy Sanjaya <contact@ardikars.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ardikars.jxpacket.core.icmp.icmp6;

import com.ardikars.jxpacket.core.icmp.Icmp;
import com.ardikars.jxpacket.core.icmp.Icmp6;

/**
 * @author Ardika Rommy Sanjaya
 * @since 1.1.5
 */
public class Icmp6DestinationUnreachable extends Icmp.IcmpTypeAndCode {

    /**
     * A Destination Unreachable message SHOULD be generated by a router, or
     * by the IPv6 layer in the originating node, in response to a packet
     * that cannot be delivered to its destination address for reasons other
     * than congestion.  (An Icmp6InverseNeighborDiscoverySolicitation message MUST NOT be generated if a
     * packet is dropped due to congestion.)
     *
     * If the reason for the failure to deliver is lack of a matching entry
     * in the forwarding node's routing table, the Code field is set to 0.
     */

    public static final Icmp6DestinationUnreachable NO_ROUTE_TO_DESTINATION =
            new Icmp6DestinationUnreachable((byte) 0, "No route to destination");

    public static final Icmp6DestinationUnreachable COMMUNICATION_WITH_DESTINATION_ADMINIS_TRATIVELY_PROHIBITED =
            new Icmp6DestinationUnreachable((byte) 1, "Communication with destination administratively prohibited");

    public static final Icmp6DestinationUnreachable BEYOND_SCOPE_OF_SOURCE_ADDRESS =
            new Icmp6DestinationUnreachable((byte) 2, "Beyond scope of source address");

    public static final Icmp6DestinationUnreachable ADDRESS_UNREACHABLE =
            new Icmp6DestinationUnreachable((byte) 3, "Address unreachable");

    public static final Icmp6DestinationUnreachable PORT_UNREACHABLE =
            new Icmp6DestinationUnreachable((byte) 4, "Address unreachable");

    public static final Icmp6DestinationUnreachable SOURCE_ADDRESS_FAILED =
            new Icmp6DestinationUnreachable((byte) 5, "Source address failed ingress/egress policy");

    public static final Icmp6DestinationUnreachable REJECT_ROUTE_TO_DESTINATION =
            new Icmp6DestinationUnreachable((byte) 6, "Reject route to destination");

    public static final Icmp6DestinationUnreachable ERROR_IN_SOURCE_ROUTING_HEADER =
            new Icmp6DestinationUnreachable((byte) 7, "Error in Source Routing HeaderAbstract");

    public Icmp6DestinationUnreachable(Byte code, String name) {
        super((byte) 1, code, name);
    }

    /**
     * Add new {@link Icmp6DestinationUnreachable} to registry.
     * @param code icmp type code.
     * @param name icmp type name.
     * @return returns {@link Icmp6DestinationUnreachable}.
     */
    public static Icmp6DestinationUnreachable register(Byte code, String name) {
        return new Icmp6DestinationUnreachable(code, name);
    }

    @Override
    public String toString() {
        return super.toString();
    }

    static {
        Icmp6.ICMP6_REGISTRY.add(NO_ROUTE_TO_DESTINATION);
        Icmp6.ICMP6_REGISTRY.add(COMMUNICATION_WITH_DESTINATION_ADMINIS_TRATIVELY_PROHIBITED);
        Icmp6.ICMP6_REGISTRY.add(BEYOND_SCOPE_OF_SOURCE_ADDRESS);
        Icmp6.ICMP6_REGISTRY.add(ADDRESS_UNREACHABLE);
        Icmp6.ICMP6_REGISTRY.add(PORT_UNREACHABLE);
        Icmp6.ICMP6_REGISTRY.add(SOURCE_ADDRESS_FAILED);
        Icmp6.ICMP6_REGISTRY.add(REJECT_ROUTE_TO_DESTINATION);
        Icmp6.ICMP6_REGISTRY.add(ERROR_IN_SOURCE_ROUTING_HEADER);
    }

}
