package com.appzilo.sdk.video;

import android.annotation.TargetApi;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;
import androidx.appcompat.widget.Toolbar;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.JavascriptInterface;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceRequest;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.appzilo.sdk.video.backend.GigsApi;
import com.appzilo.sdk.video.backend.NoticeApi;
import com.appzilo.sdk.video.backend.model.EmbedGigsResponse;
import com.appzilo.sdk.video.backend.model.NoticeResponse;
import com.appzilo.sdk.video.core.App;
import com.appzilo.sdk.video.core.BackgroundWorker;
import com.appzilo.sdk.video.core.Result;
import com.appzilo.sdk.video.player.WebViewVideoPlayer;
import com.appzilo.sdk.video.utils.SharedPreferencesUtil;

import java.util.HashMap;
import java.util.Map;

public class WebviewFragment extends Fragment implements
        SwipeRefreshLayout.OnRefreshListener,
        BackgroundWorker.Callbacks {

    public static final String URL = "url";
    private static final String OBJ_GIGS_BACKEND = "gigs_backend.video";
    private static final String TASK_REQUEST_URL = "task_request_url";

    private WebView mWebView;
    private String mUrl;
    private Toolbar mToolbar;
    private View mView;
    private boolean hideActionBar;
    private SwipeRefreshLayout mRefresh;
    private Handler mainThreadHandler;
    private SharedPreferencesUtil mPref;
    private BackgroundWorker mWorker;
    private GigsApi mGigsBackend;

    public static WebviewFragment newInstance(Bundle args) {
        WebviewFragment fragment = new WebviewFragment();
        if (args == null) {
            args = new Bundle();
        }
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Bundle args = getArguments();
        if (args != null) {
            mUrl = args.getString(URL);
        }
        mainThreadHandler = new Handler(Looper.getMainLooper());

        if(getActivity() != null){
            mPref = new SharedPreferencesUtil(getActivity());
            mWorker = new BackgroundWorker(getActivity());
            mGigsBackend = (GigsApi) mWorker.get(OBJ_GIGS_BACKEND);
            if (mGigsBackend == null) {
                mGigsBackend = new GigsApi(getContext());
                mWorker.put(OBJ_GIGS_BACKEND, mGigsBackend);
            }

        }
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        mView = inflater.inflate(R.layout.fragment_webview, container, false);
//        mToolbar = mView.findViewById(R.id.toolbar);
        mRefresh = mView.findViewById(R.id.swipeContainer);
        mRefresh.setEnabled(false);
        mRefresh.setColorSchemeResources(R.color.primary);
        mWebView = mView.findViewById(R.id.web);
        return mView;
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        WebSettings settings = mWebView.getSettings();
        if (Build.VERSION.SDK_INT >= 17) {
            settings.setMediaPlaybackRequiresUserGesture(false);
        }else {
            settings.setPluginState(WebSettings.PluginState.ON);
        }
        settings.setJavaScriptEnabled(true);
        settings.setDomStorageEnabled(true);
        settings.setAllowFileAccess(true);

        String dbm_key = mPref.getStringValue("moo_dbm", null);
        if(dbm_key != null && !dbm_key.isEmpty()){
            String dbm = mPref.getStringValue("dbm", null);
            mUrl = mUrl + "&dbm="+dbm+"&dbm_key="+dbm_key;
        }

        mWorker.executeTask(TASK_REQUEST_URL, null, this);

        mWebView.addJavascriptInterface(new OtherWebViewJavascriptInterface(), "PlayerBridge");
        mWebView.setWebViewClient(new OtherWebViewClient());
        mWebView.setWebChromeClient(new WebChromeClient());
    }

    @Override
    public void onResume() {
        super.onResume();
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                mWebView.loadUrl("javascript:playVideo()");
            }
        });
    }

    @Override
    public void onPause() {
        super.onPause();
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                mWebView.loadUrl("javascript:pauseVideo()");
            }
        });
    }

    @Override
    public void onDestroy() {
        mainThreadHandler.removeCallbacksAndMessages(null);
        super.onDestroy();
    }

    @Override
    public void onRefresh() {
        mWebView.reload();
    }

    @Override
    public Result executeTaskInBackground(String id, Bundle args) {
        if (TASK_REQUEST_URL.equals(id)) {
            return mGigsBackend.loadUrl(mUrl);
        }
        return null;
    }

    @Override
    public void onBackgroundTaskCompleted(String id, Result result) {
        if (TASK_REQUEST_URL.equals(id)) {
            if(result.isSuccess()){
                String json = (String) result.getResult();
                EmbedGigsResponse response = App.gson().fromJson(json, EmbedGigsResponse.class);
                mWebView.loadDataWithBaseURL(response.video_info.url, response.video_html, "text/html", "utf-8", null);
            }
        }
    }

    private Map<String, String> getCustomHeaders() {
        Map<String, String> headers = new HashMap<>();
        headers.put("X-Requested-With", "com.android.browser");
        NoticeResponse noticeResponse = NoticeApi.getNoticeResponse(getContext());
        if (noticeResponse != null && noticeResponse.credential != null && noticeResponse.credential.app_key != null) {
            headers.put("App-Key", noticeResponse.credential.app_key);
        }
        return headers;
    }

    private class OtherWebViewClient extends WebViewClient{
        @Override
        @TargetApi(Build.VERSION_CODES.LOLLIPOP)
        public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request)
        {
            view.loadUrl(request.getUrl().toString(), getCustomHeaders());
            return true;
        }

        @Override
        public boolean shouldOverrideUrlLoading(WebView view, String url) {
            view.loadUrl(url, getCustomHeaders());
            return true;
        }

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            super.onPageStarted(view, url, favicon);
            mRefresh.setRefreshing(true);
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            super.onPageFinished(view, url);
            mRefresh.setRefreshing(false);
            view.setVisibility(View.VISIBLE);
        }
    }

     public class OtherWebViewJavascriptInterface {

        @JavascriptInterface
        public void onFinish() {
            if(getActivity() != null){
                getActivity().finish();
            }
        }
    }
}
