package com.appzilo.sdk.video.core;

import android.content.Context;
import android.os.Build;
import android.webkit.WebSettings;
import android.webkit.WebView;

import com.appzilo.sdk.video.backend.NoticeApi;
import com.appzilo.sdk.video.backend.model.NoticeResponse;
import com.appzilo.sdk.video.utils.SharedPreferencesUtil;
import com.appzilo.sdk.video.utils.Utils;
import com.franmontiel.persistentcookiejar.PersistentCookieJar;
import com.franmontiel.persistentcookiejar.cache.SetCookieCache;
import com.franmontiel.persistentcookiejar.persistence.SharedPrefsCookiePersistor;
import com.orm.query.Condition;
import com.orm.query.Select;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.HttpCookie;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.CipherSuite;
import okhttp3.ConnectionSpec;
import okhttp3.FormBody;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import okhttp3.TlsVersion;

public class Http {
    private static final String COOKIES = "set-cookie";
    private static OkHttpClient sClient;
    private static PersistentCookieJar mCookieJar;
    private static String mAppKey;
    private static String sUserAgent;
    private static SharedPreferencesUtil mPrefUtil;
    private static ConnectionSpec spec = new ConnectionSpec.Builder(ConnectionSpec.COMPATIBLE_TLS)
            .supportsTlsExtensions(true)
            .tlsVersions(TlsVersion.TLS_1_2, TlsVersion.TLS_1_1, TlsVersion.TLS_1_0)
            .cipherSuites(
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_RC4_128_SHA,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_DHE_DSS_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_256_CBC_SHA)
            .build();

    public static void init(Context context) {
        mPrefUtil = new SharedPreferencesUtil(context);
    }

    public static Request request(Context context, String appKey) {
        mAppKey = appKey;
        return request(context);
    }

    public static Request request(Context context) {
        if (sClient == null) {
            mCookieJar = new PersistentCookieJar(new SetCookieCache(), new SharedPrefsCookiePersistor(context));
            sClient = new OkHttpClient.Builder()
                    .connectionSpecs(Collections.singletonList(spec))
                    .connectTimeout(20, TimeUnit.SECONDS)
                    .readTimeout(20, TimeUnit.SECONDS)
                    .writeTimeout(20, TimeUnit.SECONDS)
                    .cookieJar(new SyncCookieHandler())
                    .build();
        }

        return new Request(context);
    }

    public static void destroy() {
        sClient = null;
    }

    public static String getDefaultAgent(Context context) {
        if (sUserAgent == null || sUserAgent.isEmpty()) {
            // sUserAgent = System.getProperty("http.agent");
            if (Build.VERSION.SDK_INT >= 17) {
                sUserAgent = WebSettings.getDefaultUserAgent(context);
            } else {
                try {
                    Constructor<WebSettings> constructor = WebSettings.class.getDeclaredConstructor(Context.class, WebView.class);
                    constructor.setAccessible(true);
                    try {
                        WebSettings settings = constructor.newInstance(context, null);
                        sUserAgent = settings.getUserAgentString();
                    } finally {
                        constructor.setAccessible(false);
                    }
                } catch (Exception e) {
                    sUserAgent = System.getProperty("http.agent");
                }
            }
        }
        return sUserAgent;
    }


    public static class Request {
        private okhttp3.Request.Builder mBuilder;
        private List<String> mLog = new ArrayList<>();
        private Context mContext;
        private SharedPreferencesUtil mPrefUtil;
        private String mUrl;

        Request(Context context) {
            mBuilder = new okhttp3.Request.Builder();
            mPrefUtil = new SharedPreferencesUtil(context);
            NoticeResponse response = NoticeApi.getNoticeResponse(context);
            if (response != null && response.credential.app_key != null) {
                mAppKey = response.credential.app_key;
            }
            mBuilder.addHeader("App-Key", mAppKey);
            mLog.add("App-Key: " + mAppKey);
            String userAgent = Http.getDefaultAgent(context);
            if (userAgent != null) {
                mBuilder.addHeader("User-Agent", userAgent);
                mLog.add("User Agent: " + userAgent);
            }

            mContext = context;
        }

        public Request url(String url) {
            mUrl = url;
            mBuilder.url(url);
            mLog.add(0, "URL: " + url);
            return this;
        }

        public Request tag(String tag) {
            mBuilder.tag(tag);
            return this;
        }

        public Request post(Map<String, String> params) {
            if (params != null) {
                FormBody.Builder builder = new FormBody.Builder();
                if (!mPrefUtil.getStringValue("dbm", "").isEmpty()) {
                    params.put("dbm", mPrefUtil.getStringValue("dbm", ""));
                }

                if (mUrl.contains("appzilo.com") && !mPrefUtil.getStringValue("az_dbm", "").isEmpty()) {
                    params.put("dbm_key", mPrefUtil.getStringValue("az_dbm", ""));
                }

                if (mUrl.contains("getmoocash.com") && !mPrefUtil.getStringValue("moo_dbm", "").isEmpty()) {
                    params.put("dbm_key", mPrefUtil.getStringValue("moo_dbm", ""));
                }

                for (Map.Entry<String, String> entry : params.entrySet()) {
                    String value = entry.getValue();
                    if (value == null) {
                        value = "";
                    }
                    builder.add(entry.getKey(), value);
                }

                mLog.add("PARAM: " + params.toString());
                mBuilder.post(builder.build());
            }
            return this;
        }

        public Result send() {
            Result result;
            if (!Utils.isOnline(mContext) || sClient == null) {
                result = new Result(Error.OFFLINE);
            } else {
                String output = null;
                try {
                    Response response = sClient.newCall(mBuilder.build()).execute();
                    mLog.add("Code: " + response.code() + " " + response.message());
                    if (response.isSuccessful()) {
                        updateCookies(response);
                        // reading whole response as byte array (response.body().string()) may consume high memory and cause OutOfMemoryError,
                        // therefore, read the response as byte stream and convert to string using string builder
                        output = response.body().string();//sb.toString();
                    }
                } catch (IOException e) {
                    mLog.add("Error: " + e.getMessage());
                }
                if (output != null) {
                    mLog.add("Response: " + output);
                    result = new Result(null, output);
                } else {
                    result = new Result(Error.NETWORK);
                }
            }
            for (String log : mLog) {
                Logger.d("HTTP[" + Integer.toHexString(hashCode()) + "] " + log);
            }
            return result;
        }

        public Call call() {
            return sClient.newCall(mBuilder.build());
        }

        private void updateCookies(Response response) {
            List<String> setCookies = response.headers("Set-Cookie");
            StringBuilder b = new StringBuilder();
            boolean first = true;
            for (String setCookie : setCookies) {
                try {
                    List<HttpCookie> cookies = HttpCookie.parse(setCookie);
                    for (HttpCookie cookie : cookies) {
                        String name = cookie.getName();
                        String value = cookie.getValue();
                        if (first) {
                            first = false;
                        } else {
                            b.append("; ");
                        }
                        b.append(name).append("=").append(cookie.getValue());
                    }

                } catch (IllegalArgumentException ignored) {
                }
            }

            mLog.add("Update-Cookie: " + b.toString());
        }
    }
}
