package com.appzilo.sdk.video.player;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.appzilo.sdk.video.player.listeners.VideoPlayerInitListener;
import com.appzilo.sdk.video.player.listeners.VideoPlayerListener;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * WebView implementing the actual YouTube Player
 */
class WebViewVideoPlayer extends WebView implements VideoPlayer, VideoPlayerBridge.YouTubePlayerBridgeCallbacks {


    @NonNull
    private final Set<VideoPlayerListener> videoPlayerListeners;
    @NonNull
    private final Handler mainThreadHandler;
    protected boolean backgroundPlaybackEnabled = false;
    private VideoPlayerInitListener videoPlayerInitListener;
    private String baseUrl;
    private String playerHtml;

    protected WebViewVideoPlayer(Context context) {
        this(context, null);
    }

    protected WebViewVideoPlayer(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    protected WebViewVideoPlayer(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mainThreadHandler = new Handler(Looper.getMainLooper());
        videoPlayerListeners = new HashSet<>();
    }

    protected void initialize(@NonNull VideoPlayerInitListener initListener, @Nullable IFramePlayerOptions playerOptions) {
        videoPlayerInitListener = initListener;
        initWebView(playerOptions == null ? IFramePlayerOptions.getDefault() : playerOptions);
    }

    @Override
    public void onYouTubeIframeAPIReady() {
        videoPlayerInitListener.onInitSuccess(this);
    }

    @Override
    public void loadVideo(@NonNull final String videoId, final float startSeconds) {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:loadVideo('" + videoId + "', " + startSeconds + ")");
            }
        });
    }

    @Override
    public void loadChannel(@NonNull final String channelId) {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:setChannel('" + channelId + "')");
            }
        });
    }

    @Override
    public void cueVideo(@NonNull final String videoId, final float startSeconds) {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:cueVideo('" + videoId + "', " + startSeconds + ")");
            }
        });
    }

    @Override
    public void play() {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:playVideo()");
            }
        });
    }

    @Override
    public void pause() {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:pauseVideo()");
            }
        });
    }

    @Override
    public void setVolume(final int volumePercent) {
        if (volumePercent < 0 || volumePercent > 100)
            throw new IllegalArgumentException("Volume must be between 0 and 100");

        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:setVolume(" + volumePercent + ")");
            }
        });
    }

    @Override
    public void seekTo(final float time) {
        mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
                loadUrl("javascript:seekTo(" + time + ")");
            }
        });
    }

    @Override
    public void destroy() {
        videoPlayerListeners.clear();
        mainThreadHandler.removeCallbacksAndMessages(null);
        super.destroy();
    }

    @NonNull
    public Collection<VideoPlayerListener> getListeners() {
        return Collections.unmodifiableCollection(new HashSet<>(videoPlayerListeners));
    }

    @Override
    public boolean addListener(@NonNull VideoPlayerListener listener) {
        if (listener == null) {
            Log.e("YouTubePlayer", "null YouTubePlayerListener not allowed.");
            return false;
        }
        return videoPlayerListeners.add(listener);
    }

    @Override
    public boolean removeListener(@NonNull VideoPlayerListener listener) {
        return videoPlayerListeners.remove(listener);
    }

    @SuppressLint("SetJavaScriptEnabled")
    private void initWebView(IFramePlayerOptions playerOptions) {
        this.getSettings().setJavaScriptEnabled(true);
        this.getSettings().setDomStorageEnabled(true);
        this.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);
        this.getSettings().setMediaPlaybackRequiresUserGesture(false);
        setWebViewClient(new CustomWebViewClient());
        this.addJavascriptInterface(new VideoPlayerBridge(this), "PlayerBridge");
        // if the video's thumbnail is not in memory, show a black screen
        this.setWebChromeClient(new WebChromeClient() {
            @Override
            public Bitmap getDefaultVideoPoster() {
                Bitmap result = super.getDefaultVideoPoster();
                if (result == null)
                    return Bitmap.createBitmap(1, 1, Bitmap.Config.RGB_565);
                else
                    return result;
            }
        });

        if (playerHtml != null && !playerHtml.isEmpty()) {
            final String unformattedString = playerHtml;
            final String formattedString = unformattedString.replace("<<injectedPlayerVars>>", playerOptions.toString());
            this.loadDataWithBaseURL(baseUrl, formattedString, "text/html", "utf-8", null);
            return;
        } else if (baseUrl != null) {
            loadUrl(baseUrl);
        }

        if (videoPlayerInitListener != null) videoPlayerInitListener.onInitSuccess(this);

    }

    public void initWebView(String url, String html) {
        playerHtml = html;
        baseUrl = url;
        initWebView(IFramePlayerOptions.getDefault());
    }

    public void refresh() {
        playerHtml = null;
        baseUrl = null;
        loadUrl("about:blank");
    }

    private String readYouTubePlayerHTMLFromFile() {
        try {
            int raw = 0;
            InputStream inputStream = getResources().openRawResource(raw);

            InputStreamReader inputStreamReader = new InputStreamReader(inputStream, "utf-8");
            BufferedReader bufferedReader = new BufferedReader(inputStreamReader);

            String read;
            StringBuilder sb = new StringBuilder();

            while ((read = bufferedReader.readLine()) != null)
                sb.append(read).append("\n");
            inputStream.close();

            return sb.toString();
        } catch (Exception e) {
            throw new RuntimeException("Can't parse HTML file containing the player.");
        }
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        if (backgroundPlaybackEnabled && (visibility == View.GONE || visibility == INVISIBLE))
            return;

        super.onWindowVisibilityChanged(visibility);
    }

    public class CustomWebViewClient extends WebViewClient {
        boolean isError = false;

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            super.onPageStarted(view, url, favicon);
            if (!url.equals("about:blank")) {
                if (playerHtml == null || playerHtml.isEmpty()) {
//                    for (VideoPlayerListener listener : videoPlayerListeners) {
//                        listener.onReady();
//                    }
                    isError = false;
                }
            }
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            super.onPageFinished(view, url);
            if (!url.equals("about:blank")) {
                if ((playerHtml == null || playerHtml.isEmpty()) && !isError) {
                    if (view.getProgress() == 100) {
                        for (VideoPlayerListener listener : videoPlayerListeners) {
                            listener.onStateChange(PlayerConstants.PlayerState.PLAYING);
                        }
                    }
                }
                if (view.getProgress() == 100) {
                    playerHtml = null;
                    baseUrl = null;
                }
            }
        }

        @Override
        public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
            super.onReceivedError(view, request, error);
            if(playerHtml == null || playerHtml.isEmpty()){
                isError = true;
                for(VideoPlayerListener listener:videoPlayerListeners){
                    listener.onError(PlayerConstants.PlayerError.VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER);
                }
            }
        }
    }
}
