package com.appzilo.sdk.video;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.SwitchCompat;
import android.view.View;
import android.widget.Button;
import android.widget.CompoundButton;
import android.widget.TextView;

import com.appzilo.sdk.video.R;
import com.appzilo.sdk.video.backend.model.EmbedGigsResponse;
import com.appzilo.sdk.video.core.Logger;
import com.appzilo.sdk.video.player.PlayerConstants;
import com.appzilo.sdk.video.player.VideoPlayer;
import com.appzilo.sdk.video.player.VideoPlayerView;
import com.appzilo.sdk.video.player.listeners.AbstractVideoPlayerListener;
import com.appzilo.sdk.video.utils.ResourcesUtil;
import com.appzilo.sdk.video.utils.SharedPreferencesUtil;
import com.appzilo.sdk.video.utils.Utils;

public class CustomVideoUIController extends AbstractVideoPlayerListener implements
        View.OnClickListener,
        CompoundButton.OnCheckedChangeListener {

    private static final String YT_IS_AUTO_SKIP = "yt_is_auto_skip";
    private static final String ERROR_INVALID_PARAMETER_IN_REQUEST = "2";
    private static final String ERROR_HTML_5_PLAYER = "5";
    private static final String ERROR_VIDEO_NOT_FOUND = "100";
    private static final String ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER1 = "101";
    private static final String ERROR_VIDEO_LIVE_OFFLINE = "102";
    private static final String ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER2 = "150";
    private final int countdownInterval = 1000;
    private View mPanel;
    private View mLoadingBar;
    private SwitchCompat mAutoSkipToggle;
    private View mAutoSkipContainer;
    private Button mStopButton;
    private View mLiveIndicator;
    private TextView mLiveStatus;
    private Button mSkipButton;
    private SharedPreferencesUtil mPref;
    private int mDefaultInterval = 0;
    private int mDelayInterval = 10000;
    private Context context;
    private VideoPlayer youTubePlayerCallback;
    private Listener mListener;
    private int mMinWatchTime;
    private String mIdVideoPlaying;
    private boolean mPlayerIsPlaying;
    private EmbedGigsResponse mNextYtInfo;
    private int mTimestamp;
    private boolean mFirstTime;
    private Handler mHandler;
    private boolean mIsYtEnded;
    private boolean mIsPolice;
    private CountDownRunnable mCountDownRunnable;
    private boolean mIsOnPause;
    private int mCurrentSeconds;
    private VideoPlayerView mPlayerView;
    private String mVideoUniqueId;
    private boolean mIsInitialize;
    private ErrorCountDownRunnable mErrorCountdown;
    private int mErrorReportMaxTime;
    private boolean mIsVideoHtmlExist;

    public CustomVideoUIController(Context context, View customPlayerUI, VideoPlayer youTubePlayerCallback, VideoPlayerView playerView, int videoInterval, boolean isPolice) {
        this.context = context;
        if (context instanceof Listener) {
            mListener = (Listener) context;
        }
        ResourcesUtil.init(context);
        this.youTubePlayerCallback = youTubePlayerCallback;
        if (videoInterval > 0) mDelayInterval = videoInterval;
        initViews(customPlayerUI);
        mIsPolice = isPolice;
        mPref = new SharedPreferencesUtil(context);
        mPlayerView = playerView;
        mHandler = new Handler(Looper.getMainLooper());
    }

    private void initViews(View view) {
        mPanel = view.findViewById(R.id.panel);
        mLoadingBar = view.findViewById(R.id.progress_bar);
        mAutoSkipContainer = view.findViewById(R.id.yt_auto_skip_container);
        mAutoSkipToggle = view.findViewById(R.id.yt_auto_skip);
        mStopButton = view.findViewById(R.id.stop_button);
        mAutoSkipToggle.setOnCheckedChangeListener(this);
        mSkipButton = view.findViewById(R.id.skip_button);
        mLiveIndicator = view.findViewById(R.id.live_container);
        mLiveStatus = view.findViewById(R.id.live_status);
        if (mIsPolice) {
            mSkipButton.setVisibility(View.GONE);
        }
        if (mSkipButton != null) {
            mSkipButton.setOnClickListener(this);
        }

        mStopButton.setOnClickListener(this);
    }

    @Override
    public void onReady() {
        mHandler.postDelayed(new DelayRunnable(mVideoUniqueId), mDefaultInterval);
        mDefaultInterval = mDelayInterval;
    }

    @Override
    public void onStateChange(@NonNull PlayerConstants.PlayerState state) {
        if (state == PlayerConstants.PlayerState.PLAYING && mIsInitialize) {
            stopErrorReportCountDown();
            if (mCountDownRunnable == null) {
                mPlayerIsPlaying = true;
                mCountDownRunnable = new CountDownRunnable();
                mHandler.postDelayed(mCountDownRunnable, countdownInterval);
                mPanel.setBackgroundColor(ContextCompat.getColor(context, android.R.color.transparent));
                mIsInitialize = false;
            }
        }
        if (state == PlayerConstants.PlayerState.BUFFERING || state == PlayerConstants.PlayerState.VIDEO_CUED) {
            mPanel.setBackgroundColor(ContextCompat.getColor(context, android.R.color.transparent));
        } else if (state == PlayerConstants.PlayerState.ENDED) {
            mPlayerIsPlaying = false;
            if (mNextYtInfo != null) {
                initializeYtVideoInfo(mNextYtInfo);
            }
            mIsYtEnded = true;
        } else if (state == PlayerConstants.PlayerState.OFFLINE) {
            onError(PlayerConstants.PlayerError.ERROR_VIDEO_LIVE_OFFLINE);
        }
    }

    @Override
    public void onCurrentSecond(float second) {
        int roundedSecond = Math.round(second);
        onCurrentSecond(roundedSecond);
    }

    private boolean onCurrentSecond(int second) {
        int roundedSecond = Math.round(second);
        if ((roundedSecond == mMinWatchTime || roundedSecond == 0) && mFirstTime) {
            mFirstTime = false;
            mListener.videoReachEndpoint(mIdVideoPlaying, mTimestamp);
            if (!mIsPolice) mSkipButton.setVisibility(View.GONE);
            return false;
        } else if (mFirstTime && roundedSecond < mMinWatchTime) {
            if (!mIsPolice) mSkipButton.setVisibility(View.VISIBLE);
            initSkipAdCountDown(String.format(ResourcesUtil.getString(R.string.yt_skip_video_count_down), (mMinWatchTime - roundedSecond)));
            return true;
        }

        return false;
    }

    @Override
    public void onError(@NonNull PlayerConstants.PlayerError error) {
        super.onError(error);
        mPlayerIsPlaying = false;
        mDefaultInterval = 0;
        mPanel.setBackgroundColor(ResourcesUtil.getColor(R.color.black));
        mIsYtEnded = true;
        if (!mIsPolice) {
            mListener.youtubePlayableError(mIdVideoPlaying, parsePlayerError(error));
            intiUI();
            return;
        }
        mListener.videoNotAvailable();
    }

    @Override
    public void onClick(View v) {
        int i = v.getId();
        if (i == R.id.skip_button) {
            mPlayerIsPlaying = false;
            if (mNextYtInfo != null) initializeYtVideoInfo(mNextYtInfo);
        } else if (i == R.id.stop_button) {
            mListener.forceStopVideo();
        }
    }

    @Override
    public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
        int i = buttonView.getId();
        if (i == R.id.yt_auto_skip) {
            mAutoSkipToggle.setChecked(isChecked);
            mPref.putBooleanValue(YT_IS_AUTO_SKIP, isChecked);
        }
    }

    public void initializeYtVideoInfo(EmbedGigsResponse response) {
        if (!mPlayerIsPlaying) {
            youTubePlayerCallback.pause();
            mPlayerView.refreshPlayer();
            mPanel.setBackgroundColor(ResourcesUtil.getColor(R.color.black));
            if (response.success) {
                intiUI();
                mPanel.setClickable(!mIsPolice && !response.enable_touch);
                mPanel.setFocusable(!mIsPolice && !response.enable_touch);
                mCountDownRunnable = null;
                mNextYtInfo = null;
                mCurrentSeconds = 0;
                mMinWatchTime = response.video_info.min_watch_time;
                mVideoUniqueId = response.video_info.id;
                mIdVideoPlaying = response.gid;
                mTimestamp = response.ts;
                mErrorReportMaxTime = response.video_info.max_error_time;
                mIsVideoHtmlExist = response.video_html != null && !response.video_html.isEmpty();

                if (response.video_html != null && !response.video_html.isEmpty()) {

                    mPlayerView.initPlayer(response.video_info.url, response.video_html);
                } else {
                    mHandler.postDelayed(new ViewDelayRunnable(response), mDefaultInterval);
                }

            } else {
                mListener.videoNotAvailable();
            }
        }
    }

    private void createErrorReportCountdown() {
        if(mErrorReportMaxTime > 0 && mIsVideoHtmlExist){
            Logger.e("createErrorReportCountdown:");
            if(mErrorCountdown != null) mHandler.removeCallbacks(mErrorCountdown);
            mErrorCountdown = new ErrorCountDownRunnable();

            mHandler.postDelayed(mErrorCountdown, mErrorReportMaxTime * 1000);
        }
    }

    private void stopErrorReportCountDown(){
        if(mErrorCountdown != null){
            Logger.e("stopErrorReportCountDown:");
            mHandler.removeCallbacks(mErrorCountdown);
            mErrorCountdown = null;
        }
    }

    private void intiUI() {
        mLoadingBar.setVisibility(View.VISIBLE);
        mAutoSkipContainer.setVisibility(View.VISIBLE);
        mStopButton.setVisibility(View.VISIBLE);
        mAutoSkipToggle.setChecked(mPref.getBooleanValue(YT_IS_AUTO_SKIP, false));
        if (!mIsPolice) mSkipButton.setVisibility(View.GONE);
    }

    public void onResume() {
        if (mPlayerIsPlaying && this.youTubePlayerCallback != null) {
            this.youTubePlayerCallback.play();
        }
        if (mIsOnPause && mHandler != null && mCountDownRunnable != null) {
            mHandler.postDelayed(mCountDownRunnable, countdownInterval);
        }
        mIsOnPause = false;
        mStopButton.setOnClickListener(this);
    }

    public void onPause() {
        mIsOnPause = true;
        mStopButton.setOnClickListener(null);
        if (this.youTubePlayerCallback != null) {
            this.youTubePlayerCallback.pause();
        }
        stopErrorReportCountDown();

    }

    public void onDestroy() {
        if (mHandler != null) {
            mHandler.removeCallbacksAndMessages(null);
        }
        mAutoSkipToggle.setOnCheckedChangeListener(null);

    }

    public void verifiedYtVideo(EmbedGigsResponse response) {
        mNextYtInfo = response;
        initSkipAd();
        if (mPref.getBooleanValue(YT_IS_AUTO_SKIP, false) || mIsYtEnded) {
            mPlayerIsPlaying = false;
            initializeYtVideoInfo(response);
        }
    }

    private void initSkipAdCountDown(String content) {
        if (mSkipButton != null) {
            mSkipButton.setText(content);
            mSkipButton.setTextSize(12);
            int paddingInPx = Utils.dpToPx(8);
            mSkipButton.setPadding(paddingInPx, paddingInPx, paddingInPx, paddingInPx);
            mSkipButton.setEnabled(false);
        }
    }

    private void initSkipAd() {
        if (mSkipButton != null) {
            mSkipButton.setText(R.string.yt_skip_video);
            int paddingInPx = Utils.dpToPx(8);
            mSkipButton.setPadding(paddingInPx, paddingInPx, paddingInPx, paddingInPx);
            mSkipButton.setEnabled(true);
            if (!mIsPolice) mSkipButton.setVisibility(View.VISIBLE);
        }
    }

    private void showUiVideo() {
        mFirstTime = true;
        mIsYtEnded = false;
        mNextYtInfo = null;
        mLoadingBar.setVisibility(View.GONE);
        mAutoSkipContainer.setVisibility(View.GONE);
        mStopButton.setVisibility(View.GONE);
    }

    @NonNull
    private String parsePlayerError(PlayerConstants.PlayerError error) {
        String playerError;
        if (error.equals(PlayerConstants.PlayerError.INVALID_PARAMETER_IN_REQUEST))
            playerError = ERROR_INVALID_PARAMETER_IN_REQUEST;
        else if (error.equals(PlayerConstants.PlayerError.HTML_5_PLAYER))
            playerError = ERROR_HTML_5_PLAYER;
        else if (error.equals(PlayerConstants.PlayerError.VIDEO_NOT_FOUND))
            playerError = ERROR_VIDEO_NOT_FOUND;
        else if (error.equals(PlayerConstants.PlayerError.VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER))
            playerError = ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER1;
        else if (error.equals(PlayerConstants.PlayerError.VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER))
            playerError = ERROR_VIDEO_NOT_PLAYABLE_IN_EMBEDDED_PLAYER2;
        else if (error.equals(PlayerConstants.PlayerError.ERROR_VIDEO_LIVE_OFFLINE))
            playerError = ERROR_VIDEO_LIVE_OFFLINE;
        else
            playerError = "0";
        return playerError;
    }

    public interface Listener {
        void videoReachEndpoint(String id, int ts);

        void videoNotAvailable();

        void forceStopVideo();

        void youtubePlayableError(String id, String error);
    }

    class ErrorCountDownRunnable implements Runnable {

        @Override
        public void run() {
            onError(PlayerConstants.PlayerError.UNKNOWN);
        }
    }

    class CountDownRunnable implements Runnable {

        @Override
        public void run() {
            if(mIsOnPause){
                return;
            }
            if(mMinWatchTime > 0) mCurrentSeconds++;
            if(onCurrentSecond(mCurrentSeconds)){
                mHandler.postDelayed(mCountDownRunnable, countdownInterval);
            }
        }
    }

    class ViewDelayRunnable implements Runnable {

        private EmbedGigsResponse response;

        ViewDelayRunnable(EmbedGigsResponse response) {
            this.response = response;
        }

        @Override
        public void run() {
            showUiVideo();
            mIsInitialize = true;
            mPlayerView.initPlayer(response.video_info.url, response.video_html);
            mDefaultInterval = mDelayInterval;
        }
    }

    class DelayRunnable implements Runnable {

        private String mVideoId;

        DelayRunnable(String videoId) {
            mVideoId = videoId;
        }

        @Override
        public void run() {
            showUiVideo();
            createErrorReportCountdown();
            mIsInitialize = true;
            youTubePlayerCallback.loadVideo(mVideoId, 0);
            mDefaultInterval = mDelayInterval;
        }
    }
}
