package com.appzilo.sdk.video.player;

import android.arch.lifecycle.Lifecycle;
import android.arch.lifecycle.LifecycleObserver;
import android.arch.lifecycle.OnLifecycleEvent;
import android.content.Context;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebView;
import android.widget.FrameLayout;

import com.appzilo.sdk.video.player.listeners.AbstractVideoPlayerListener;
import com.appzilo.sdk.video.player.listeners.VideoPlayerFullScreenListener;
import com.appzilo.sdk.video.player.listeners.VideoPlayerInitListener;
import com.appzilo.sdk.video.player.playerUtils.FullScreenHelper;
import com.appzilo.sdk.video.player.playerUtils.PlaybackResumer;
import com.appzilo.sdk.video.ui.DefaultPlayerUIController;
import com.appzilo.sdk.video.ui.PlayerUIController;
import com.appzilo.sdk.video.utils.Callable;
import com.appzilo.sdk.video.utils.NetworkReceiver;
import com.appzilo.sdk.video.utils.Utils;


public class VideoPlayerView extends FrameLayout implements NetworkReceiver.NetworkListener, LifecycleObserver {
    public static final String PLAYER_TYPE = "player_type";
    public static final int TYPE_YOUTUBE = 0;
    public static final int TYPE_TWITCH = 1;
    public static final int TYPE_MIXER = 2;

    @NonNull
    private final WebViewVideoPlayer youTubePlayer;
    @NonNull
    private final NetworkReceiver networkReceiver;
    @NonNull
    private final PlaybackResumer playbackResumer;
    @NonNull
    private final FullScreenHelper fullScreenHelper;
    @Nullable
    private DefaultPlayerUIController defaultPlayerUIController;
    @Nullable
    private Callable asyncInitialization;

    public VideoPlayerView(Context context) {
        this(context, null);
    }

    public VideoPlayerView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public VideoPlayerView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        youTubePlayer = new WebViewVideoPlayer(context);
        LayoutParams frameLayout = new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        addView(youTubePlayer, frameLayout);

        defaultPlayerUIController = new DefaultPlayerUIController(this, youTubePlayer);

        playbackResumer = new PlaybackResumer();
        networkReceiver = new NetworkReceiver(this);
        fullScreenHelper = new FullScreenHelper();

        fullScreenHelper.addFullScreenListener(defaultPlayerUIController);
        addYouTubePlayerListeners(youTubePlayer);
    }

    public void initPlayer(String url, String html) {
        youTubePlayer.initWebView(url, html);
    }

    public void refreshPlayer() {
        youTubePlayer.refresh();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        // if height == wrap content make the view 16:9
        if (getLayoutParams().height == ViewGroup.LayoutParams.WRAP_CONTENT) {
            int sixteenNineHeight = MeasureSpec.makeMeasureSpec(MeasureSpec.getSize(widthMeasureSpec) * 9 / 16, MeasureSpec.EXACTLY);
            super.onMeasure(widthMeasureSpec, sixteenNineHeight);
        } else
            super.onMeasure(widthMeasureSpec, heightMeasureSpec);
    }

    /**
     * Initialize the player
     *
     * @see VideoPlayerView#initialize(VideoPlayerInitListener, boolean, IFramePlayerOptions)
     */
    public void initialize(@NonNull final VideoPlayerInitListener videoPlayerInitListener, boolean handleNetworkEvents) {
        initialize(videoPlayerInitListener, handleNetworkEvents, null);
    }

    /**
     * Initialize the player
     *
     * @param videoPlayerInitListener listener for player init events
     * @param handleNetworkEvents     if true a broadcast receiver will be registered. If false you should handle network events with your own broadcast receiver.
     * @param playerOptions           customizable options for the embedded video player
     */
    public void initialize(@NonNull final VideoPlayerInitListener videoPlayerInitListener, boolean handleNetworkEvents, @Nullable final IFramePlayerOptions playerOptions) {
        if (handleNetworkEvents)
            getContext().registerReceiver(networkReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));

        asyncInitialization = new Callable() {
            @Override
            public void call() {
                youTubePlayer.initialize(new VideoPlayerInitListener() {
                    @Override
                    public void onInitSuccess(@NonNull VideoPlayer videoPlayer) {
                        videoPlayerInitListener.onInitSuccess(videoPlayer);
                    }
                }, playerOptions);
            }
        };

        if (Utils.isOnline(getContext()))
            asyncInitialization.call();
    }

    /**
     * Calls {@link WebView#destroy()} on the player. And unregisters the broadcast receiver (for network events), if registered.
     * Call this method before destroying the host Fragment/Activity, or register this View as an observer of its host lifcecycle
     */
    @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
    public void release() {
        removeView(youTubePlayer);
        youTubePlayer.removeAllViews();
        youTubePlayer.destroy();
        try {
            getContext().unregisterReceiver(networkReceiver);
        } catch (Exception ignore) {
        }
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
    void onStop() {
        youTubePlayer.pause();
    }

    @Override
    public void onNetworkAvailable() {
        if (asyncInitialization != null)
            asyncInitialization.call();
        else
            playbackResumer.resume(youTubePlayer);
    }

    @Override
    public void onNetworkUnavailable() {
    }

    public void enableBackgroundPlayback(boolean enable) {
        youTubePlayer.backgroundPlaybackEnabled = enable;
    }

    @NonNull
    public PlayerUIController getPlayerUIController() {
        if (defaultPlayerUIController == null)
            throw new RuntimeException("You have inflated a custom player UI. You must manage it with your own controller.");
        return defaultPlayerUIController;
    }

    /**
     * Replaces the default UI of the player with a custom UI.
     * You will have to control the custom UI in your application,
     * the default controller obtained through {@link VideoPlayerView#getPlayerUIController()} won't be available anymore.
     *
     * @param customPlayerUILayoutID the ID of the layout defining the custom UI.
     * @return The inflated View
     */
    public View inflateCustomPlayerUI(@LayoutRes int customPlayerUILayoutID) {
        removeViews(1, this.getChildCount() - 1);

        if (defaultPlayerUIController != null) {
            youTubePlayer.removeListener(defaultPlayerUIController);
            fullScreenHelper.removeFullScreenListener(defaultPlayerUIController);
        }

        defaultPlayerUIController = null;

        return View.inflate(getContext(), customPlayerUILayoutID, this);
    }

    public void enterFullScreen() {
        fullScreenHelper.enterFullScreen(this);
    }

    public void exitFullScreen() {
        fullScreenHelper.exitFullScreen(this);
    }

    public boolean isFullScreen() {
        return fullScreenHelper.isFullScreen();
    }

    public void toggleFullScreen() {
        fullScreenHelper.toggleFullScreen(this);
    }

    public boolean addFullScreenListener(@NonNull VideoPlayerFullScreenListener fullScreenListener) {
        return fullScreenHelper.addFullScreenListener(fullScreenListener);
    }

    public boolean removeFullScreenListener(@NonNull VideoPlayerFullScreenListener fullScreenListener) {
        return fullScreenHelper.removeFullScreenListener(fullScreenListener);
    }

    private void addYouTubePlayerListeners(final VideoPlayer videoPlayer) {
        if (defaultPlayerUIController != null) videoPlayer.addListener(defaultPlayerUIController);
        videoPlayer.addListener(playbackResumer);
        videoPlayer.addListener(new AbstractVideoPlayerListener() {
            @Override
            public void onReady() {
                asyncInitialization = null;
            }
        });
    }
}
