@file:Suppress("unused")

package com.appspiriment.permissionutils

import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import kotlinx.coroutines.*


/*********************************************************
 * Class   :  PermissionManager
 * Author  :  Arun Nair
 * Created :  14/10/20
 *******************************************************
 * Purpose :
 *******************************************************
 * Rework Details:
 * 1) {Author} :  {Date} : {Details}
 *********************************************************/

object PermissionManager {
    private const val TAG = "PermissionManager"
    private val parentJob = Job()
    private val coroutineScope = CoroutineScope(parentJob + Dispatchers.Default)

    /*********************************
     * Create a static coroutineScope
     * and return [CoroutineScope]
     ********************************/
    fun getCoroutineScope(): CoroutineScope {
        return coroutineScope
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissions : Permissions VarArg
     ******************************************/
    suspend fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return requestPermissionsFromUser(
            supportFragmentManager,
            requestId,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissions : Permissions VarArg
     ******************************************/
    suspend fun Fragment.requestPermissionsAsync(
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return requestPermissionsFromUser(
            childFragmentManager,
            requestId,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionGrantedListener
     * @permissionDeniedListener
     * @permissionDeniedPermanently
     * @permissionRationalListener
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit,
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionRationalListener: (requestId: Int) -> Unit,
        requestPermissionOnRationale: Boolean,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                supportFragmentManager,
                requestId,
                permissionGrantedListener,
                permissionDeniedListener,
                permissionDeniedPermanently,
                permissionRationalListener,
                requestPermissionOnRationale,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionGrantedListener
     * @permissionDeniedListener
     * @permissionDeniedPermanently
     * @permissionRationalListener
     * @permissions
     ******************************************/
    fun Fragment.requestPermissionsAsync(
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit,
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionRationalListener: (requestId: Int) -> Unit,
        requestPermissionOnRationale: Boolean,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                childFragmentManager,
                requestId,
                permissionGrantedListener,
                permissionDeniedListener,
                permissionDeniedPermanently,
                permissionRationalListener,
                requestPermissionOnRationale,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListner
     * @permissions
     ******************************************/
    fun AppCompatActivity.requestPermissionsAsync(
        requestId: Int,
        permissionListner: PermissionListener,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                supportFragmentManager,
                requestId,
                permissionListner,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *
     * @requestId : Request Id
     * @permissionListner
     * @permissions
     ******************************************/
    fun Fragment.requestPermissionsAsync(
        requestId: Int,
        permissionListner: PermissionListener,
        vararg permissions: String
    ) {
        getCoroutineScope().launch {
            requestPermissionsFromUser(
                childFragmentManager,
                requestId,
                permissionListner,
                *permissions
            )
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *******************************************/
    private suspend fun requestPermissionsFromUser(
        fragmentManager: FragmentManager,
        requestId: Int,
        permissionListner: PermissionListener,
        vararg permissions: String
    ) {
        requestPermissionsFromUser(
            fragmentManager, requestId,
            permissionListner::onPermissionGranted,
            permissionListner::onPermissionDenied,
            permissionListner::onPermissionDeniedPermanently,
            permissionListner::onShowRational,
            false,
            *permissions
        )
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *******************************************/
    private suspend fun requestPermissionsFromUser(
        fragmentManager: FragmentManager,
        requestId: Int,
        permissionGrantedListener: (requestId: Int) -> Unit,
        permissionDeniedListener: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionDeniedPermanently: (requestId: Int, deniedPermissions: List<String>) -> Unit,
        permissionRationalListener: (requestId: Int) -> Unit,
        requestPermissionOnRationale: Boolean = false,
        vararg permissions: String
    ) {

        requestPermissionsFromUser(fragmentManager, requestId, *permissions).let {
           MainScope().launch {
               when (it) {
                   is PermissionResult.PermissionGranted -> permissionGrantedListener(it.requestCode)
                   is PermissionResult.PermissionDenied -> permissionDeniedListener(
                       it.requestCode,
                       it.deniedPermissions
                   )
                   is PermissionResult.PermissionDeniedPermanently -> permissionDeniedPermanently(
                       it.requestCode,
                       it.permanentlyDeniedPermissions
                   )
                   is PermissionResult.ShowRational -> {
                       if (requestPermissionOnRationale) requestPermissionsFromUser(
                           fragmentManager, requestId,
                           permissionGrantedListener,
                           permissionDeniedListener,
                           permissionDeniedPermanently,
                           permissionRationalListener,
                           false,
                           *permissions
                       ) else permissionRationalListener(it.requestCode)
                   }
               }
           }
        }
    }

    /******************************************
     * Request for permissions asynchronously
     * and return [PermissionResult]
     *******************************************/
    private suspend fun requestPermissionsFromUser(
        fragmentManager: FragmentManager,
        requestId: Int,
        vararg permissions: String
    ): PermissionResult {
        return withContext(Dispatchers.Main) {
            val permissionFragment =
                fragmentManager.findFragmentByTag(TAG) ?: PermissionFragment().apply {
                    fragmentManager.beginTransaction().add(this, TAG).commitNow()
                }

            permissionFragment.let {
                (it as PermissionFragment).run {
                    deferredPermission = CompletableDeferred()
                    requestPermissionsFromUser(requestId, *permissions)
                    deferredPermission.await()
                }
            }
        }
    }
}
