package com.appspiriment.baseclasses

import android.os.Bundle
import android.view.*
import androidx.activity.OnBackPressedCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.fragment.app.Fragment
import androidx.lifecycle.LiveData
import androidx.lifecycle.Observer
import androidx.navigation.NavDirections
import androidx.navigation.fragment.findNavController
import com.appspiriment.baseclasses.utils.BaseConstants
import com.appspiriment.baseclasses.utils.BaseViewStates
import org.koin.androidx.viewmodel.ext.android.viewModel
import kotlin.reflect.KClass

abstract class BaseFragment<
        out ViewModelType : BaseViewModel,
        out DataBindingType : ViewDataBinding>(
    viewModelClass: KClass<ViewModelType>,
    val layoutId: Int,
    val viewModelVarId: Int,
    val menuId: Int = R.menu.basedefaultmenu,
    val isHomeFragment:Boolean = false,
    val homeFragmentId: Int = BaseConstants.HOME_DEFAULT_ID,
    val isBackpressHandled: Boolean = true,
    val isHomeAsUpEnabled: Boolean = false
) : Fragment() {

    /***************************************
     * Declarations
     ***************************************/
    val viewModel by viewModel(viewModelClass)
    protected var menu: Menu? = null

    /***************************************
     * Setting Observers
     ***************************************/
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {

        val binding =
            DataBindingUtil.inflate(inflater, layoutId, container, false) as DataBindingType
        binding.setVariable(BR.viewModel, viewModel)
        binding.lifecycleOwner = this

        setHasOptionsMenu((menuId != R.menu.basedefaultmenu))
        retainInstance = true

        return binding.root
    }


    /**
     * *************************************
     * OnOptionsMenu Create
     * **************************************
     */
    override fun onCreateOptionsMenu(menu: Menu, inflater: MenuInflater) {
        super.onCreateOptionsMenu(menu, inflater)
        inflater.inflate(menuId, menu)
        this.menu = menu
    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return if (item.itemId == android.R.id.home) {
            navigateBack()
        } else {
            handleOptionsItemClick(item.itemId)
        }
    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        try {
            (activity as AppCompatActivity).supportActionBar?.setDisplayHomeAsUpEnabled(
                isHomeAsUpEnabled
            )
        } catch (ignore: Exception) {
        }

        initializeViews()

        setBaseObservers()

        initializeAfterObserving()

    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    override fun onActivityCreated(savedInstanceState: Bundle?) {
        super.onActivityCreated(savedInstanceState)
        if (isBackpressHandled) {
            requireActivity()
                .onBackPressedDispatcher
                .addCallback(viewLifecycleOwner,
                    object : OnBackPressedCallback(true) {
                        override fun handleOnBackPressed() {
                            onBackpress()
                        }
                    })
        }
    }

    /***************************************
     * Setting Observers
     ***************************************/
    open fun <T> setObserver(liveData: LiveData<T>, vararg functions: (value: T) -> Unit) {
        liveData.observe(viewLifecycleOwner, Observer { value ->
            for (function in functions)
                function(value)
        })
    }

    /***************************************
     * Setting Observers
     ***************************************/
    private fun setBaseObservers() {
        setObserver(viewModel.viewState, { state ->
            run {
                when (state) {
                    BaseViewStates.STATE_FINISH -> activity?.finish()
                    BaseViewStates.STATE_FINISH_WITH_DIALOG ->
                        showConfirmFinishActivityDialog()
                    else -> observeViewState(state)
                }

                if (state != BaseViewStates.STATE_NONE)
                    viewModel.setViewState(BaseViewStates.STATE_NONE)
            }
        })

        setObserver(viewModel.navigate, { viewId ->
            if (viewId != BaseViewStates.STATE_NONE) {
                navigateToClickedDestination(viewId)
                viewModel.setNavigate(BaseViewStates.STATE_NONE)
            }
        })

        setViewObservers()
    }


    /***************************************
     * Setting Observers
     ***************************************/
    open fun navigateToClickedDestination(viewId: Int) {
        getNavDirectionsAction(viewId)?.let {
            findNavController().navigate(it)
        }
    }

    /***************************************
     * Setting Observers
     ***************************************/
    open fun getNavDirectionsAction(viewId: Int): NavDirections? {
        return null
    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    open fun onBackpress() {
        navigateBack()
    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    open fun navigateBack(): Boolean {
       return if (!isHomeFragment) {
           getNavigateToHomeArgs()?.let {
               findNavController().navigate(homeFragmentId, it)
           } ?: findNavController().navigate(homeFragmentId)
           return true
        } else false
    }

    /***************************************
     * Setting Observers
     ***************************************/
    open fun observeViewState(state: Int) {}

    open fun showConfirmFinishActivityDialog() {}
    open fun getNavigateToHomeArgs(): Bundle? = null
    open fun setViewObservers() {}
    open fun initializeAfterObserving() {}
    open fun initializeViews() {}
    open fun handleOptionsItemClick(menuItemId: Int) = false
}