// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.streams;

import java.util.Collections;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class CollectorUtils {

	static final Set<Collector.Characteristics> CH_NOID = Collections.emptySet();

	public static <T> Collector<T, ?, T> single() {
		return new CollectorImpl<T, NonUniqueResultCheckingConsumer<T>, T>(NonUniqueResultCheckingConsumer::new, NonUniqueResultCheckingConsumer::accept, null,
				new NoResultCheckingFunction<T>(), CH_NOID);
	}

	public static <T> Collector<T, ?, T> singleOrNull() {
		return new CollectorImpl<T, NonUniqueResultCheckingConsumer<T>, T>(NonUniqueResultCheckingConsumer::new, NonUniqueResultCheckingConsumer::accept, null, f -> f.result,
				CH_NOID);
	}

	public static <T> Collector<T, ?, T> firstOrNull() {
		return new CollectorImpl<T, FirstResultConsumer<T>, T>(FirstResultConsumer::new, FirstResultConsumer::accept, null, f -> f.result, CH_NOID);
	}

	static class NonUniqueResultCheckingConsumer<T> implements Consumer<T> {
		T result = null;

		@Override
		public void accept(T obj) {
			if (this.result != null) {
				throw new NonUniqueResultException();
			} else {
				this.result = obj;
			}
		}
	}

	static class NoResultCheckingFunction<T> implements Function<NonUniqueResultCheckingConsumer<T>, T> {

		@Override
		public T apply(NonUniqueResultCheckingConsumer<T> t) {
			if (t.result == null) {
				throw new NoResultException();
			}
			return t.result;
		}
	}

	static class FirstResultConsumer<T> implements Consumer<T> {
		T result = null;

		@Override
		public void accept(T obj) {
			if (this.result == null) {
				this.result = obj;
			}
		}
	}

	static class CollectorImpl<T, A, R> implements Collector<T, A, R> {
		final Supplier<A> supplier;
		final BiConsumer<A, T> accumulator;
		final BinaryOperator<A> combiner;
		final Function<A, R> finisher;
		final Set<Characteristics> characteristics;

		CollectorImpl(Supplier<A> supplier, BiConsumer<A, T> accumulator, BinaryOperator<A> combiner, Function<A, R> finisher, Set<Characteristics> characteristics) {
			this.supplier = supplier;
			this.accumulator = accumulator;
			this.combiner = combiner;
			this.finisher = finisher;
			this.characteristics = characteristics;
		}

		@Override
		public BiConsumer<A, T> accumulator() {
			return this.accumulator;
		}

		@Override
		public Supplier<A> supplier() {
			return this.supplier;
		}

		@Override
		public BinaryOperator<A> combiner() {
			return this.combiner;
		}

		@Override
		public Function<A, R> finisher() {
			return this.finisher;
		}

		@Override
		public Set<Characteristics> characteristics() {
			return this.characteristics;
		}
	}
}
