// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.mail;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;

import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.MimeMessage;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.crypto.SecureProps;
import com.appslandia.common.threading.ThreadTask;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.CollectionUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class SmtpMailer extends InitializeObject {

	protected SecureProps props;
	protected Session session;

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.props, "props is required.");

		this.session = Session.getInstance(this.props);
	}

	public void send(MailMessage message) throws MessagingException {
		send(CollectionUtils.toList(new ArrayList<>(1), message));
	}

	public void send(List<MailMessage> messages) throws MessagingException {
		initialize();
		AssertUtils.assertNotNull(messages);

		try (Transport transport = this.session.getTransport("smtp")) {

			String user = AssertUtils.assertNotNull(this.props.get("mail.smtp.user"), "mail.smtp.user is required.");
			String password = AssertUtils.assertNotNull(this.props.get("mail.smtp.password"), "mail.smtp.password is required.");
			transport.connect(user, password);

			for (MailMessage mailMessage : messages) {
				MimeMessage message = mailMessage.toMimeMessage(this);
				message.saveChanges();

				transport.sendMessage(message, message.getAllRecipients());
			}
		}
	}

	public void sendAsync(MailMessage message, Executor executor) {
		sendAsync(CollectionUtils.toList(new ArrayList<>(1), message), executor);
	}

	public void sendAsync(List<MailMessage> messages, Executor executor) {
		initialize();
		AssertUtils.assertNotNull(messages);
		AssertUtils.assertNotNull(executor);

		executor.execute(new ThreadTask() {

			@Override
			protected void doRun() throws Exception {
				send(messages);
			}
		});
	}

	public SmtpMailer setProps(SecureProps props) {
		assertNotInitialized();

		if (props != null) {
			this.props = props.copy();
		}
		return this;
	}
}
