// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.util.regex.Pattern;

import com.appslandia.common.utils.DateUtils;
import com.appslandia.common.utils.StringUtils;
import com.appslandia.common.utils.URLEncoding;

/**
 * 
 * 
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ValuesEncoder {
	private static final String NULL_VAL = "null";

	final StringBuilder sb = new StringBuilder();

	protected void addVal(String encodedVal) {
		if (this.sb.length() > 0) {
			this.sb.append(',');
		}
		this.sb.append(encodedVal);
	}

	public ValuesEncoder add(String value) {
		addVal(!StringUtils.isNullOrEmpty(value) ? URLEncoding.encodeParam(value) : NULL_VAL);
		return this;
	}

	public ValuesEncoder add(boolean value) {
		addVal(Boolean.toString(value));
		return this;
	}

	public ValuesEncoder add(int value) {
		addVal(Integer.toString(value));
		return this;
	}

	public ValuesEncoder add(long value) {
		addVal(Long.toString(value));
		return this;
	}

	public ValuesEncoder add(float value) {
		addVal(Float.toString(value));
		return this;
	}

	public ValuesEncoder add(double value) {
		addVal(Double.toString(value));
		return this;
	}

	public ValuesEncoder add(Object value) {
		addVal(toStringValue(value));
		return this;
	}

	public String toEncoded() {
		String values = this.sb.toString();
		return !values.isEmpty() ? values : null;
	}

	public static String[] parse(String encoded) throws IllegalArgumentException {
		if (StringUtils.isNullOrEmpty(encoded)) {
			return StringUtils.EMPTY_ARRAY;
		}
		String[] values = encoded.split(Pattern.quote(","));
		for (int i = 0; i < values.length; i++) {
			values[i] = !NULL_VAL.equals(values[i]) ? URLEncoding.decodeParam(values[i]) : null;
		}
		return values;
	}

	private static String toStringValue(Object value) {
		if (value == null) {
			return null;
		}
		if (value instanceof java.sql.Date) {
			return DateUtils.iso8601Date((java.sql.Date) value);
		}
		if (value instanceof java.sql.Time) {
			return DateUtils.iso8601Time((java.sql.Time) value);
		}
		if (value instanceof java.sql.Timestamp) {
			return DateUtils.iso8601DateTime((java.sql.Timestamp) value);
		}
		return value.toString();
	}
}
