// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.lang.annotation.Annotation;
import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.net.URI;
import java.net.URL;
import java.nio.Buffer;
import java.nio.charset.Charset;
import java.security.ProtectionDomain;
import java.util.BitSet;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.logging.Logger;

import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.ExceptionUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.ReflectionUtils;
import com.appslandia.common.utils.TypeUtils;
import com.appslandia.common.utils.ValueUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ToStringBuilder {

	@Target({ ElementType.TYPE, ElementType.FIELD })
	@Retention(RetentionPolicy.RUNTIME)
	@Documented
	public @interface ToObjectInfo {
	}

	@Target({ ElementType.FIELD })
	@Retention(RetentionPolicy.RUNTIME)
	@Documented
	public @interface Compact {
	}

	public abstract static class FieldDecision {

		static boolean isJdkClass(Class<?> type) {
			// @formatter:off
			return	type.getName().startsWith("java.") || 
					type.getName().startsWith("javax.") ||
					type.getName().startsWith("jakarta.") ||
					type.getName().startsWith("sun.") ||
					type.getName().startsWith("com.sun.") || 
					type.getName().startsWith("com.oracle.") || 
					type.getName().startsWith("jdk.") ||
					type.getName().startsWith("org.omg.") || 
					type.getName().startsWith("org.w3c.");
			// @formatter:on
		}

		public boolean isToObjectInfo(Object value, Field member) {
			if (member != null) {
				if (member.getAnnotation(ToObjectInfo.class) != null) {
					return true;
				}
			}
			if (!isJdkClass(value.getClass())) {
				if (ReflectionUtils.findAnnotation(value.getClass(), ToObjectInfo.class) != null) {
					return true;
				}
			}
			return useToObjectInfo(value, member);
		}

		protected abstract boolean useToObjectInfo(Object value, Field member);
	}

	public static final FieldDecision DEFAULT_DECISION = new FieldDecision() {

		@Override
		protected boolean useToObjectInfo(Object value, Field member) {
			return false;
		}
	};

	private int level;
	private boolean isCompact;
	private boolean tabAsIdent;
	private FieldDecision fieldDecision = DEFAULT_DECISION;

	private int identTabs;

	public ToStringBuilder() {
		this(2);
	}

	public ToStringBuilder(int level) {
		setLevel(level);
	}

	public ToStringBuilder fieldDecision(FieldDecision fieldDecision) {
		this.fieldDecision = fieldDecision;
		return this;
	}

	public String toString(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs, this.isCompact);
		if (obj == null) {
			return builder.append("null").toString();
		}
		this.toStringObject(obj, 1, this.isCompact, builder);
		return builder.toString();
	}

	public String toStringFields(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs, this.isCompact);
		if (obj == null) {
			return builder.append("null").toString();
		}
		this.toStringFields(obj, 1, this.isCompact, builder);
		return builder.toString();
	}

	private void toStringObject(Object obj, int level, boolean isCompact, TextBuilder builder) {
		if (obj == null) {
			builder.append("null");
			return;
		}
		if (obj instanceof Iterable) {
			toStringIterator(obj, new IteratorIterator(((Iterable<?>) obj).iterator()), level, isCompact, builder);
			return;
		}
		if (obj instanceof Iterator) {
			toStringIterator(obj, new IteratorIterator((Iterator<?>) obj), level, isCompact, builder);
			return;
		}
		if (obj instanceof Enumeration) {
			toStringIterator(obj, new EnumerationIterator((Enumeration<?>) obj), level, isCompact, builder);
			return;
		}
		if (obj.getClass().isArray()) {
			boolean isPrimOrWrapper = TypeUtils.isPrimOrWrapperType(obj.getClass().getComponentType());
			toStringIterator(obj, new ArrayIterator(obj, isPrimOrWrapper), level, isPrimOrWrapper, builder);
			return;
		}
		if (obj instanceof Buffer) {
			Object arr = ((Buffer) obj).array();
			boolean isPrimOrWrapper = TypeUtils.isPrimOrWrapperType(arr.getClass().getComponentType());
			toStringIterator(obj, new ArrayIterator(arr, isPrimOrWrapper), level, isPrimOrWrapper, builder);
			return;
		}
		if (obj instanceof Map) {
			toStringMap((Map<?, ?>) obj, level, isCompact, builder);
			return;
		}
		if (obj instanceof Throwable) {
			builder.append(ExceptionUtils.toStackTrace((Throwable) obj));
			return;
		}
		if (obj.getClass() == String.class) {
			builder.append("\"").append(obj).append("\"").append(" (").append(((String) obj).length()).append(")");
			return;
		}
		if (obj instanceof CharSequence) {
			builder.append(ObjectUtils.toObjectInfo(obj)).append(": ").append("\"").append(obj).append("\"").append(" (").append(obj.toString().length()).append(")");
			return;
		}
		if (obj instanceof BitSet) {
			builder.append(ObjectUtils.toObjectInfo(obj)).append(": ").append(obj);
			return;
		}
		if (TypeUtils.isPrimOrWrapperType(obj.getClass())) {
			builder.append(obj);
			return;
		}
		if (useToString(obj.getClass())) {
			builder.append(obj);
		} else {
			toStringFields(obj, level, isCompact, builder);
		}
	}

	private void toStringFields(Object obj, int level, boolean isCompact, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(obj));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		Class<?> clazz = obj.getClass();
		while (clazz != null) {
			Field[] fields = clazz.getDeclaredFields();
			for (Field field : fields) {
				if (!isFirst) {
					builder.append(",");
				} else {
					isFirst = false;
				}

				appendln(builder, isCompact);
				appendtab(builder, level + this.identTabs, isCompact);
				builder.append(field.getName()).append(": ");

				try {
					field.setAccessible(true);
					Object fieldVal = field.get(obj);

					if (fieldVal == null) {
						builder.append("null");
					} else {
						if (this.fieldDecision.isToObjectInfo(fieldVal, field)) {
							toObjectInfo(builder, fieldVal, field);
						} else {
							boolean compact = isFieldCompact(fieldVal, field, isCompact);
							this.toStringObject(fieldVal, level + 1, compact, builder);
						}
					}

				} catch (Throwable ex) {
					builder.append("error=").append(ExceptionUtils.buildMessage(ex));
				}
			}
			clazz = clazz.getSuperclass();
		}

		if (isFirst) {
			builder.append(" no fields ]");
		} else {
			appendln(builder, isCompact);
			appendtab(builder, level - 1 + this.identTabs, isCompact).append("]");
		}
	}

	private void toStringIterator(Object obj, ElementIterator iterator, int level, boolean isCompact, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(obj));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		while (iterator.hasNext()) {
			Object element = iterator.next();

			if (element == MORE_ELEMENTS) {
				builder.append(", AND MORE ...");
				break;
			}

			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder, isCompact);
			appendtab(builder, level + this.identTabs, isCompact);

			if (element == null) {
				builder.append("null");
			} else {
				if (this.fieldDecision.isToObjectInfo(element, null)) {
					builder.append(ObjectUtils.toObjectInfo(element));
				} else {
					this.toStringObject(element, level + 1, isCompact, builder);
				}
			}
		}
		if (isFirst) {
			builder.append(" no elements ]");
		} else {
			appendln(builder, isCompact);
			appendtab(builder, level - 1 + this.identTabs, isCompact).append("] (").append(iterator.getComputedLen()).append(")");
		}
	}

	private void toStringMap(Map<?, ?> map, int level, boolean isCompact, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(map));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		for (Object key : map.keySet()) {
			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder, isCompact);
			appendtab(builder, level + this.identTabs, isCompact);

			builder.append(key).append(": ");
			Object entryVal = map.get(key);
			if (entryVal == null) {
				builder.append("null");
			} else {
				if (this.fieldDecision.isToObjectInfo(entryVal, null)) {
					builder.append(ObjectUtils.toObjectInfo(entryVal));
				} else {
					this.toStringObject(entryVal, level + 1, isCompact, builder);
				}
			}
		}

		if (isFirst) {
			builder.append(" no entries ]");
		} else {
			appendln(builder, isCompact);
			appendtab(builder, level - 1 + this.identTabs, isCompact).append("] (").append(map.size()).append(")");
		}
	}

	private void toStringAttributes(Object obj, Method getAttributeMethod, Set<String> attributes, int level, boolean isCompact, TextBuilder builder) {
		builder.append("[");
		boolean isFirst = true;

		for (String attribute : attributes) {
			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder, isCompact);
			appendtab(builder, level + this.identTabs, isCompact);
			builder.append(attribute).append(": ");

			try {
				Object element = getAttributeMethod.invoke(obj, attribute);
				if (element == null) {
					builder.append("null");
				} else {
					if ("javax.servlet.error.exception".equals(attribute)) {
						builder.append(ObjectUtils.toObjectInfo(element));
					} else if (this.fieldDecision.isToObjectInfo(element, null)) {
						builder.append(ObjectUtils.toObjectInfo(element));
					} else {
						this.toStringObject(element, level + 1, isCompact, builder);
					}
				}

			} catch (Throwable ex) {
				builder.append("error=").append(ExceptionUtils.buildMessage(ex));
			}
		}

		if (isFirst) {
			builder.append(" no elements ]");
		} else {
			appendln(builder, isCompact);
			appendtab(builder, level - 1 + this.identTabs, isCompact).append("]");
		}
	}

	public String toStringAttributes(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs, this.isCompact);
		if (obj == null) {
			builder.append("null");
			return builder.toString();
		}
		try {
			Set<String> attributes = getAttributeNames(obj, "getAttributeNames");
			Method method = ReflectionUtils.findMethod(obj.getClass(), "getAttribute");
			AssertUtils.assertNotNull(method);

			builder.append(ObjectUtils.toObjectInfo(obj)).append("-attributes");
			toStringAttributes(obj, method, attributes, 1, this.isCompact, builder);
		} catch (Throwable ex) {
			builder.append("error=").append(ExceptionUtils.buildMessage(ex));
		}
		return builder.toString();
	}

	public String toStringHeaders(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs, this.isCompact);
		if (obj == null) {
			builder.append("null");
			return builder.toString();
		}
		try {
			Set<String> attributes = getAttributeNames(obj, "getHeaderNames");
			Method method = ReflectionUtils.findMethod(obj.getClass(), "getHeaders");
			AssertUtils.assertNotNull(method);

			builder.append(ObjectUtils.toObjectInfo(obj)).append("-headers");
			toStringAttributes(obj, method, attributes, 1, this.isCompact, builder);
		} catch (Throwable ex) {
			builder.append("error=").append(ExceptionUtils.buildMessage(ex));
		}
		return builder.toString();
	}

	private static Set<String> getAttributeNames(Object obj, String methodName) throws Exception {
		Method method = ReflectionUtils.findMethod(obj.getClass(), methodName);
		AssertUtils.assertNotNull(method);

		Object attrs = method.invoke(obj);
		Set<String> names = new TreeSet<>();
		if (attrs instanceof Enumeration) {

			Enumeration<String> enm = ObjectUtils.cast(attrs);
			while (enm.hasMoreElements()) {
				names.add(enm.nextElement());
			}
		} else {
			Collection<String> attrCol = ObjectUtils.cast(attrs);
			names.addAll(attrCol);
		}
		return names;
	}

	public ToStringBuilder setLevel(int level) {
		this.level = ValueUtils.valueOrMin(level, 1);
		return this;
	}

	public ToStringBuilder useCompact() {
		this.isCompact = true;
		return this;
	}

	public ToStringBuilder useTabAsIdent() {
		this.tabAsIdent = true;
		return this;
	}

	public ToStringBuilder setIdentTabs(int identTabs) {
		this.identTabs = ValueUtils.valueOrMin(identTabs, 0);
		return this;
	}

	private TextBuilder appendln(TextBuilder builder, boolean isCompact) {
		if (!isCompact) {
			builder.appendln();
		}
		return builder;
	}

	private TextBuilder appendtab(TextBuilder builder, int tabs, boolean isCompact) {
		if (isCompact) {
			builder.appendsp();

		} else if (this.tabAsIdent) {
			builder.appendtab(tabs);
		} else {
			builder.append4sp(tabs);
		}
		return builder;
	}

	protected boolean useToString(Class<?> type) {
		if (Enum.class.isAssignableFrom(type)) {
			return true;
		}
		if (Date.class.isAssignableFrom(type) || Calendar.class.isAssignableFrom(type) || TimeZone.class.isAssignableFrom(type) || (type == Locale.class)
				|| Charset.class.isAssignableFrom(type)) {
			return true;
		}
		// JDK8+ Date/Time objects
		if (type.getPackage().getName().equals("java.time")) {
			return true;
		}
		if ((type == File.class) || type.getName().equals("java.nio.file.Path") || (type == URL.class) || (type == URI.class)) {
			return true;
		}
		if (AnnotatedElement.class.isAssignableFrom(type) || Type.class.isAssignableFrom(type) || ClassLoader.class.isAssignableFrom(type)) {
			return true;
		}
		if (type == ProtectionDomain.class) {
			return true;
		}
		if (Annotation.class.isAssignableFrom(type)) {
			return true;
		}
		if (InputStream.class.isAssignableFrom(type) || OutputStream.class.isAssignableFrom(type)) {
			return true;
		}
		if (Reader.class.isAssignableFrom(type) || Writer.class.isAssignableFrom(type)) {
			return true;
		}
		if (ResourceBundle.class.isAssignableFrom(type) || Logger.class.isAssignableFrom(type)) {
			return true;
		}
		return false;
	}

	protected boolean isFieldCompact(Object fieldVal, Field field, boolean isCompact) {
		if (isCompact || (field.getAnnotation(Compact.class) != null)) {
			return true;
		}
		if (field.getType().isArray()) {
			Class<?> elementType = field.getType().getComponentType();
			if ((elementType == String.class) || TypeUtils.isPrimOrWrapperType(elementType) || (elementType == BigDecimal.class)) {
				if (Array.getLength(fieldVal) <= 512) {
					return true;
				}
			}
		}
		return false;
	}

	static void toObjectInfo(TextBuilder builder, Object fieldVal, Field member) {
		builder.append(member.getType().getName());
		if (!member.getType().isPrimitive()) {
			builder.append("@").append(Integer.toHexString(fieldVal.hashCode()));
		}
	}

	interface ElementIterator {

		boolean hasNext();

		Object next();

		int getIndex();

		int getComputedLen();
	}

	static final Object MORE_ELEMENTS = new Object() {
	};

	static class ArrayIterator implements ElementIterator {
		static final int MAX_BYTE_CHAR_PRINTED = 4096;

		final Object obj;
		final int len;
		final boolean isPrimOrWrapper;
		int index = 0;

		public ArrayIterator(Object obj, boolean isPrimOrWrapper) {
			this.obj = obj;
			this.len = Array.getLength(obj);
			this.isPrimOrWrapper = isPrimOrWrapper;
		}

		@Override
		public boolean hasNext() {
			return this.index < this.len;
		}

		@Override
		public Object next() {
			if (!this.isPrimOrWrapper) {
				return Array.get(this.obj, this.index++);
			}
			if (this.index < MAX_BYTE_CHAR_PRINTED) {
				return Array.get(this.obj, this.index++);
			} else {
				return MORE_ELEMENTS;
			}
		}

		@Override
		public int getIndex() {
			return this.index;
		}

		@Override
		public int getComputedLen() {
			return this.len;
		}
	}

	static class IteratorIterator implements ElementIterator {
		final Iterator<?> obj;
		int index = 0;

		public IteratorIterator(Iterator<?> obj) {
			this.obj = obj;
		}

		@Override
		public boolean hasNext() {
			return this.obj.hasNext();
		}

		@Override
		public Object next() {
			this.index++;
			return this.obj.next();
		}

		@Override
		public int getIndex() {
			return this.index;
		}

		@Override
		public int getComputedLen() {
			return this.index;
		}
	}

	static class EnumerationIterator implements ElementIterator {
		final Enumeration<?> obj;
		int index = 0;

		public EnumerationIterator(Enumeration<?> obj) {
			this.obj = obj;
		}

		@Override
		public boolean hasNext() {
			return this.obj.hasMoreElements();
		}

		@Override
		public Object next() {
			this.index++;
			return this.obj.nextElement();
		}

		@Override
		public int getIndex() {
			return this.index;
		}

		@Override
		public int getComputedLen() {
			return this.index;
		}
	}
}
