// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import java.text.Normalizer;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class NormalizeUtils {

	private static final Pattern STRIP_ACCENTS_PATTERN = Pattern.compile("\\p{InCombiningDiacriticalMarks}+");

	private static final Map<Character, Character> STRIP_ACCENTS_MAP;

	static {
		Map<Character, Character> map = new HashMap<>();
		map.put('\u0110', 'D');
		map.put('\u0111', 'd');

		map.put('\u0126', 'H');
		map.put('\u0127', 'h');

		map.put('\u0141', 'L');
		map.put('\u0142', 'l');

		map.put('\u0166', 'T');
		map.put('\u0167', 't');
		STRIP_ACCENTS_MAP = Collections.unmodifiableMap(map);
	}

	public static String unaccent(String text) {
		if (text == null) {
			return null;
		}

		StringBuilder sb = new StringBuilder(Normalizer.normalize(text, Normalizer.Form.NFD));
		for (int i = 0; i < sb.length(); i++) {
			Character chr = STRIP_ACCENTS_MAP.get(text.charAt(i));
			if (chr != null) {
				sb.setCharAt(i, chr);
			}
		}
		return STRIP_ACCENTS_PATTERN.matcher(sb).replaceAll(StringUtils.EMPTY_STRING);
	}

	private static final Pattern[] WHITESPACE_PUNCT_DOUBLE_HYPHEN_PATTERNS = PatternUtils.compile("\\s+|\\p{Punct}+", "-{2,}");

	public static String toLabel(String text) {
		return normalize(text, WHITESPACE_PUNCT_DOUBLE_HYPHEN_PATTERNS, '-');
	}

	private static final Pattern[] DOUBLE_WHITESPACES_PATTERNS = PatternUtils.compile("\\s{2,}");

	public static String toText(String text) {
		return normalize(text, DOUBLE_WHITESPACES_PATTERNS, (char) 32);
	}

	public static String normalize(String text, Pattern[] replacePatterns, char bySep) {
		if (text == null) {
			return null;
		}
		String replacement = new String(new char[] { bySep });
		for (Pattern p : replacePatterns) {
			text = p.matcher(text).replaceAll(replacement);
		}
		text = trim(text, bySep);
		return !text.isEmpty() ? text : null;
	}

	private static String trim(String name, char charToTrim) {
		int start = -1;
		while ((++start < name.length()) && (name.charAt(start) == charToTrim)) {
		}

		int end = name.length();
		while ((--end >= 0) && (name.charAt(end) == charToTrim)) {
		}
		if (start >= end) {
			return StringUtils.EMPTY_STRING;
		}
		return name.substring(start, end + 1);
	}
}
