// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.jose;

import java.io.StringReader;
import java.util.HashSet;
import java.util.Set;

import com.appslandia.common.base.BaseEncoder;
import com.appslandia.common.base.DestroyException;
import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.base.StringWriter;
import com.appslandia.common.crypto.CryptoException;
import com.appslandia.common.json.JsonException;
import com.appslandia.common.json.JsonProcessor;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.CharsetUtils;
import com.appslandia.common.utils.CollectionUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.ValueUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class JwtProcessor extends InitializeObject {

	protected JsonProcessor jsonProcessor;
	protected Set<String> numericDateProps;
	protected JoseSigner joseSigner;

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.jsonProcessor, "jsonProcessor is required.");

		if (this.numericDateProps == null) {
			this.numericDateProps = CollectionUtils.unmodifiableSet(JwtPayload.EXP, JwtPayload.NBF, JwtPayload.IAT);
		} else {
			CollectionUtils.unmodifiableSet(this.numericDateProps, JwtPayload.EXP, JwtPayload.NBF, JwtPayload.IAT);
		}

		this.joseSigner = ValueUtils.valueOrDefault(this.joseSigner, JoseSigner.NONE);
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.jsonProcessor != null) {
			this.jsonProcessor.destroy();
		}
		if (this.joseSigner != null) {
			this.joseSigner.destroy();
		}
	}

	protected void verifyHeader(JoseHeader header, boolean parsed) throws JoseException {
		// ALG
		if (header.getAlgorithm() == null) {
			throw new JoseException("alg is required.");
		}
		if (!header.getAlgorithm().equals(this.joseSigner.getAlgorithm())) {
			throw new JoseException("alg is not matched.");
		}
		// KID
		if (!ObjectUtils.equals(header.getKid(), this.joseSigner.getKid())) {
			throw new JoseException("kid is not matched.");
		}
	}

	public String build(JwtObject jwt) throws CryptoException, JsonException, JoseException {
		this.initialize();
		AssertUtils.assertNotNull(jwt);
		AssertUtils.assertNotNull(jwt.getHeader());
		AssertUtils.assertNotNull(jwt.getPayload());

		verifyHeader(jwt.getHeader(), false);

		StringWriter out = new StringWriter();
		this.jsonProcessor.write(out, jwt.getHeader());
		String base64Header = BaseEncoder.BASE64_URL_NP.encode(out.toString().getBytes(CharsetUtils.UTF_8));

		out.reset();
		this.jsonProcessor.write(out, jwt.getPayload());
		String base64Payload = BaseEncoder.BASE64_URL_NP.encode(out.toString().getBytes(CharsetUtils.UTF_8));

		// No ALG
		if (this.joseSigner == JoseSigner.NONE) {
			return base64Header + JoseUtils.JOSE_PART_SEP + base64Payload + JoseUtils.JOSE_PART_SEP;
		}

		// ALG
		String dataToSign = base64Header + JoseUtils.JOSE_PART_SEP + base64Payload;
		String base64Sig = BaseEncoder.BASE64_URL_NP.encode(this.joseSigner.sign(dataToSign.getBytes(CharsetUtils.UTF_8)));
		return dataToSign + JoseUtils.JOSE_PART_SEP + base64Sig;
	}

	public JwtObject parse(String jwt) throws CryptoException, JsonException, JoseException {
		this.initialize();
		AssertUtils.assertNotNull(jwt);

		String[] parts = JoseUtils.parseParts(jwt);
		if (parts == null) {
			throw new JoseException("jwt is invalid.");
		}

		// No ALG
		if (parts[2] == null) {
			if (this.joseSigner != JoseSigner.NONE) {
				throw new JoseException("Signature is required.");
			}
			return doParse(parts[0], parts[1], null);
		}

		// ALG
		String dataToSign = parts[0] + JoseUtils.JOSE_PART_SEP + parts[1];
		byte[] signature = BaseEncoder.BASE64_URL_NP.decode(parts[2]);

		if (!this.joseSigner.verify(dataToSign.getBytes(CharsetUtils.UTF_8), signature)) {
			throw new JoseException("Failed to verify signature.");
		}
		return doParse(parts[0], parts[1], parts[2]);
	}

	protected JwtObject doParse(String headerPart, String payloadPart, String signature) throws JsonException, JoseException {
		String headerJson = new String(BaseEncoder.BASE64_URL_NP.decode(headerPart), CharsetUtils.UTF_8);
		String payloadJson = new String(BaseEncoder.BASE64_URL_NP.decode(payloadPart), CharsetUtils.UTF_8);

		JoseHeader header = new JoseHeader(this.jsonProcessor.readAsLinkedMap(new StringReader(headerJson)));
		JoseUtils.convertToNumericDates(header, this.numericDateProps);

		JwtPayload payload = new JwtPayload(this.jsonProcessor.readAsLinkedMap(new StringReader(payloadJson)));
		JoseUtils.convertToNumericDates(payload, this.numericDateProps);

		verifyHeader(header, true);
		return new JwtObject(header, payload, signature);
	}

	public JsonProcessor getJsonProcessor() {
		this.initialize();
		return this.jsonProcessor;
	}

	public JwtProcessor setJsonProcessor(JsonProcessor jsonProcessor) {
		assertNotInitialized();
		this.jsonProcessor = jsonProcessor;
		return this;
	}

	public Set<String> getNumericDateProps() {
		this.initialize();
		return this.numericDateProps;
	}

	public JwtProcessor setNumericDateProps(String... numericDateProps) {
		assertNotInitialized();
		if (numericDateProps != null) {
			this.numericDateProps = CollectionUtils.toSet(numericDateProps);
		}
		return this;
	}

	public JoseSigner getJoseSigner() {
		this.initialize();
		return this.joseSigner;
	}

	public JwtProcessor setJoseSigner(JoseSigner joseSigner) {
		assertNotInitialized();
		this.joseSigner = joseSigner;
		return this;
	}

	public JwtProcessor copy() {
		JwtProcessor impl = new JwtProcessor();
		if (this.jsonProcessor != null) {
			impl.jsonProcessor = this.jsonProcessor.copy();
		}
		if (this.numericDateProps != null) {
			impl.numericDateProps = new HashSet<>(this.numericDateProps);
		}
		if (this.joseSigner != null) {
			impl.joseSigner = this.joseSigner.copy();
		}
		return impl;
	}
}
