// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.jose;

import java.util.Collections;
import java.util.Date;
import java.util.List;

import com.appslandia.common.utils.CollectionUtils;
import com.appslandia.common.utils.DateUtils;
import com.appslandia.common.utils.ObjectUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class JwtPayloadVerifier extends JoseVerifier<JwtPayload> {

	private String issuer;
	private List<String> audiences;
	private long leewayMs;

	@Override
	protected void init() throws Exception {
		this.audiences = CollectionUtils.unmodifiableList(this.audiences);

		// issuer
		super.addDelegate(new JoseVerifier.Delegate<JwtPayload, String>(this.issuer) {

			@Override
			public void verify(JwtPayload obj) throws JoseException {
				if (!ObjectUtils.equals(obj.getIssuer(), this.arg)) {
					throw new JoseException("issuer is not matched.");
				}
			}
		});

		// audiences
		super.addDelegate(new JoseVerifier.Delegate<JwtPayload, List<String>>(this.audiences) {

			@Override
			public void verify(JwtPayload obj) throws JoseException {
				List<String> payloadAudiences = obj.getAudiences();
				if (payloadAudiences == null) {
					payloadAudiences = Collections.emptyList();
				}

				if ((payloadAudiences.size() > this.arg.size()) || !this.arg.containsAll(payloadAudiences)) {
					throw new JoseException("audience is not matched.");
				}
			}
		});

		// IssuedAt
		super.addDelegate(new JoseVerifier.Delegate<JwtPayload, Long>(this.leewayMs) {

			@Override
			public void verify(JwtPayload obj) throws JoseException {
				Date iat = obj.getIssuedAt();
				if (iat == null) {
					return;
				}
				if (!DateUtils.isPastTime(iat.getTime(), this.arg)) {
					throw new JoseException("IssuedAt is not matched.");
				}
			}
		});

		// NBF
		super.addDelegate(new JoseVerifier.Delegate<JwtPayload, Long>(this.leewayMs) {

			@Override
			public void verify(JwtPayload obj) throws JoseException {
				Date nbf = obj.getNotBefore();
				if (nbf == null) {
					return;
				}
				if (!DateUtils.isPastTime(nbf.getTime(), this.arg)) {
					throw new JoseException("jwt can't be used before " + DateUtils.iso8601DateTime(nbf));
				}
			}
		});

		// EXP
		super.addDelegate(new JoseVerifier.Delegate<JwtPayload, Long>(this.leewayMs) {

			@Override
			public void verify(JwtPayload obj) throws JoseException {
				Date expAt = obj.getExpiresAt();
				if (expAt == null) {
					return;
				}
				if (!DateUtils.isFutureTime(expAt.getTime(), this.arg)) {
					throw new JoseException("jwt can't be used after " + DateUtils.iso8601DateTime(expAt));
				}
			}
		});
	}

	@Override
	public <A> JwtPayloadVerifier addDelegate(Delegate<JwtPayload, A> delegate) {
		super.addDelegate(delegate);
		return this;
	}

	public JwtPayloadVerifier setIssuer(String issuer) {
		assertNotInitialized();
		this.issuer = issuer;
		return this;
	}

	public JwtPayloadVerifier setAudiences(String... audiences) {
		assertNotInitialized();
		if (audiences != null) {
			this.audiences = CollectionUtils.toList(audiences);
		}
		return this;
	}

	public JwtPayloadVerifier setLeewayMs(long leewayMs) {
		assertNotInitialized();
		this.leewayMs = leewayMs;
		return this;
	}
}
