// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.jdbc;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.AssertUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class Sql extends InitializeObject implements Serializable {
	private static final long serialVersionUID = 1L;

	public static final int DEFAULT_ARRAY_MAX_LENGTH = 32;

	private String sql;
	private Map<String, Integer> arrayLens;

	private String translatedSql;
	private Map<String, int[]> indexesMap;

	public Sql() {
	}

	public Sql(String sql) {
		this.sql = sql;
	}

	public Sql arrayLen(String parameterName, int maxLength) {
		assertNotInitialized();
		AssertUtils.assertTrue(maxLength > 0, "maxLength is required.");

		if (this.arrayLens == null) {
			this.arrayLens = new HashMap<>();
		}
		this.arrayLens.put(parameterName, maxLength);
		return this;
	}

	public Sql sql(String sql) {
		assertNotInitialized();
		this.sql = sql;
		return this;
	}

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.sql, "sql is required.");
		translateSql();
	}

	private void translateSql() {
		StringBuilder sb = new StringBuilder(this.sql);
		Map<String, int[]> indexesMap = new LinkedHashMap<>();

		int i = 0;
		int index = 0;
		while (true) {
			int j = i;
			while (j < sb.length() - 1 && sb.charAt(j) != '@') {
				j++;
			}

			// @parameter
			if (j < sb.length() - 1 && Character.isLetter(sb.charAt(j + 1))) {
				int k = j + 1;
				while (k < sb.length() && (Character.isLetterOrDigit(sb.charAt(k)) || sb.charAt(k) == '_')) {
					k++;
				}
				String paramName = sb.substring(j + 1, k);
				sb.delete(j, k);

				boolean isArrayParam = isArrayContext(sb, j);
				Integer arrayLen = (this.arrayLens == null) ? null : this.arrayLens.get(paramName);
				if (arrayLen != null) {
					if (!isArrayParam) {
						throw new IllegalArgumentException("Array parameter is not found (name=" + paramName + ")");
					}
				} else {
					if (isArrayParam) {
						arrayLen = DEFAULT_ARRAY_MAX_LENGTH;
						if (this.arrayLens == null) {
							this.arrayLens = new HashMap<>();
						}
						this.arrayLens.put(paramName, arrayLen);
					}
				}
				int len = isArrayParam ? arrayLen : 1;

				// ?
				for (int subIdx = 0; subIdx < len; subIdx++) {
					if (subIdx == 0) {
						sb.insert(j, '?');
					} else {
						sb.insert(j, "?, ");
					}
				}

				// indexes
				for (int subIdx = 0; subIdx < len; subIdx++) {
					String name = isArrayParam ? toParamName(paramName, subIdx) : paramName;
					int[] indexes = indexesMap.get(name);

					if (indexes == null) {
						indexes = new int[] { ++index };
					} else {
						indexes = Arrays.copyOf(indexes, indexes.length + 1);
						indexes[indexes.length - 1] = ++index;
					}
					indexesMap.put(name, indexes);
				}

				i = j + len;
			} else if (j < sb.length() - 1) {
				throw new IllegalArgumentException("sql is invalid.");
			} else {
				break;
			}
		}

		this.translatedSql = sb.toString();
		this.indexesMap = !indexesMap.isEmpty() ? indexesMap : Collections.emptyMap();
	}

	public String getSql() {
		initialize();
		return this.sql;
	}

	public String getTranslatedSql() {
		initialize();
		return this.translatedSql;
	}

	public int[] getIndexes(String parameterName) {
		initialize();
		int[] indexes = this.indexesMap.get(parameterName);
		if (indexes == null) {
			throw new IllegalArgumentException("Parameter is required (name=" + parameterName + ")");
		}
		return indexes;
	}

	public int getArrayLen(String parameterName) {
		initialize();
		Integer len = (this.arrayLens != null) ? this.arrayLens.get(parameterName) : null;
		if (len == null) {
			throw new IllegalArgumentException("Array parameter is not found (name=" + parameterName + ")");
		}
		return len;
	}

	public static String toParamName(String parameterName, int subIdx) {
		return parameterName + "__" + subIdx;
	}

	private static boolean isArrayContext(StringBuilder sb, int paramIndex) {
		if (sb.length() < 6) {
			return false;
		}
		// )
		int k = paramIndex;
		while (k < sb.length() && Character.isWhitespace(sb.charAt(k))) {
			k++;
		}
		if ((k >= sb.length()) || (sb.charAt(k) != ')')) {
			return false;
		}

		// (
		k = paramIndex - 1;
		while (k >= 0 && Character.isWhitespace(sb.charAt(k))) {
			k--;
		}
		if ((k < 0) || (sb.charAt(k) != '(')) {
			return false;
		}

		// IN
		k = k - 1;
		while (k >= 0 && Character.isWhitespace(sb.charAt(k))) {
			k--;
		}
		if ((k <= 2) || (Character.toUpperCase(sb.charAt(k)) != 'N') || (Character.toUpperCase(sb.charAt(k - 1)) != 'I') || !Character.isWhitespace(sb.charAt(k - 2))) {
			return false;
		}
		return true;
	}
}
