// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.formatters;

import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalAccessor;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.appslandia.common.base.FormatProvider;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public abstract class Jdk8DateFormatter<T extends TemporalAccessor> implements Formatter {

	private static final ConcurrentMap<String, DateTimeFormatter> FORMATTER_CACHE = new ConcurrentHashMap<>();

	protected abstract String getPattern(FormatProvider formatProvider);

	protected abstract T parse(String str, DateTimeFormatter formatter) throws DateTimeParseException;

	@Override
	public final String format(Object obj, FormatProvider formatProvider) {
		if (obj == null) {
			return null;
		}
		return getFormatter(getPattern(formatProvider)).format((TemporalAccessor) getArgumentType().cast(obj));
	}

	@Override
	public T parse(String str, FormatProvider formatProvider) throws FormatterException {
		if (str == null) {
			return null;
		}

		try {
			return parse(str, getFormatter(getPattern(formatProvider)));

		} catch (DateTimeParseException ex) {
			throw new FormatterException("An error occurred while parsing temporal (str=" + str + ", type=" + getArgumentType() + ")", getErrorMsgKey());
		}
	}

	protected static DateTimeFormatter getFormatter(String pattern) {
		return FORMATTER_CACHE.computeIfAbsent(pattern, p -> DateTimeFormatter.ofPattern(p));
	}
}
