// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import com.appslandia.common.base.PropertyConfig;
import com.appslandia.common.utils.DateUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class SecureConfig extends PropertyConfig {
	private static final long serialVersionUID = 1L;

	final TextEncryptor textEncryptor;

	public SecureConfig(char[] password) {
		this.textEncryptor = new TextEncryptor(new PbeEncryptor().setTransformation("AES/CBC/PKCS5Padding").setKeySize(16).setPassword(password));
	}

	public SecureConfig(Encryptor encryptor) {
		this.textEncryptor = new TextEncryptor(encryptor);
	}

	public SecureConfig(TextEncryptor textEncryptor) {
		this.textEncryptor = textEncryptor;
	}

	@Override
	public SecureConfig load(InputStream is) throws IOException {
		super.load(is);
		return this;
	}

	@Override
	public SecureConfig load(Reader r) throws IOException {
		super.load(r);
		return this;
	}

	@Override
	public SecureConfig load(File file) throws IOException {
		super.load(file);
		return this;
	}

	@Override
	public SecureConfig load(String file) throws IOException {
		super.load(file);
		return this;
	}

	@Override
	public String getString(String key) throws CryptoException {
		String value = super.getString(key);
		if (value == null) {
			return null;
		}
		if (!isEncryptedValue(value)) {
			return value;
		}
		return this.textEncryptor.decrypt(parseEncryptedValue(value));
	}

	@Override
	public SecureConfig put(String key, String value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, boolean value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, int value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, long value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, float value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, double value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, Boolean value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, Integer value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, Long value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, Float value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, Double value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, java.sql.Date value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, java.sql.Time value) {
		super.put(key, value);
		return this;
	}

	@Override
	public SecureConfig put(String key, java.sql.Timestamp value) {
		super.put(key, value);
		return this;
	}

	protected void putsString(String key, String value) throws CryptoException {
		this.props.put(key, (value != null) ? markEncryptedValue(this.textEncryptor.encrypt(value)) : null);
	}

	public SecureConfig puts(String key, String value) throws CryptoException {
		putsString(key, StringUtils.trimToNull(value));
		return this;
	}

	public SecureConfig puts(String key, boolean value) throws CryptoException {
		putsString(key, Boolean.toString(value));
		return this;
	}

	public SecureConfig puts(String key, int value) throws CryptoException {
		putsString(key, Integer.toString(value));
		return this;
	}

	public SecureConfig puts(String key, long value) throws CryptoException {
		putsString(key, Long.toString(value));
		return this;
	}

	public SecureConfig puts(String key, float value) throws CryptoException {
		putsString(key, Float.toString(value));
		return this;
	}

	public SecureConfig puts(String key, double value) throws CryptoException {
		putsString(key, Double.toString(value));
		return this;
	}

	public SecureConfig puts(String key, Boolean value) throws CryptoException {
		putsString(key, (value != null) ? value.toString() : null);
		return this;
	}

	public SecureConfig puts(String key, Integer value) throws CryptoException {
		putsString(key, (value != null) ? value.toString() : null);
		return this;
	}

	public SecureConfig puts(String key, Long value) throws CryptoException {
		putsString(key, (value != null) ? value.toString() : null);
		return this;
	}

	public SecureConfig puts(String key, Float value) throws CryptoException {
		putsString(key, (value != null) ? value.toString() : null);
		return this;
	}

	public SecureConfig puts(String key, Double value) throws CryptoException {
		putsString(key, (value != null) ? value.toString() : null);
		return this;
	}

	public SecureConfig puts(String key, java.sql.Date value) throws CryptoException {
		putsString(key, (value != null) ? DateUtils.iso8601Date(value) : null);
		return this;
	}

	public SecureConfig puts(String key, java.sql.Time value) throws CryptoException {
		putsString(key, (value != null) ? DateUtils.iso8601Time(value) : null);
		return this;
	}

	public SecureConfig puts(String key, java.sql.Timestamp value) throws CryptoException {
		putsString(key, (value != null) ? DateUtils.iso8601DateTime(value) : null);
		return this;
	}

	private static final String ENC_BEGIN = "ENC(";
	private static final String ENC_END = ")";

	private static String markEncryptedValue(String value) {
		return ENC_BEGIN + value + ENC_END;
	}

	private static boolean isEncryptedValue(String value) {
		return value.startsWith(ENC_BEGIN) && value.endsWith(ENC_END);
	}

	private static String parseEncryptedValue(String value) {
		return value.substring(ENC_BEGIN.length(), value.length() - ENC_END.length());
	}
}
