// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Random;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import com.appslandia.common.base.DestroyException;
import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.ArrayUtils;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.CharsetUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class MacDigester extends InitializeObject implements Digester {
	private String algorithm, provider;
	private Mac mac;
	private byte[] secret;
	private SecretKeySpec keySpec;

	final Object mutex = new Object();
	final Random random = new SecureRandom();

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.algorithm, "algorithm is required.");
		AssertUtils.assertNotNull(this.secret, "secret is required.");

		// MAC
		if (this.provider == null) {
			this.mac = Mac.getInstance(this.algorithm);
		} else {
			this.mac = Mac.getInstance(this.algorithm, this.provider);
		}

		this.keySpec = new SecretKeySpec(this.secret, this.algorithm);
		CryptoUtils.clear(this.secret);
	}

	@Override
	public int getDigestLength() {
		this.initialize();
		return this.mac.getMacLength();
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.keySpec != null) {
			CryptoUtils.tryDestroy(this.keySpec);
		}
	}

	@Override
	public byte[] digest(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		try {
			synchronized (this.mutex) {
				this.mac.init(this.keySpec);
				return this.mac.doFinal(message);
			}
		} catch (GeneralSecurityException ex) {
			throw new CryptoException(ex);
		}
	}

	@Override
	public boolean verify(byte[] message, byte[] mac) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");
		AssertUtils.assertNotNull(mac, "mac is required.");

		byte[] msgMac = null;
		try {
			synchronized (this.mutex) {
				this.mac.init(this.keySpec);
				msgMac = this.mac.doFinal(message);
			}
		} catch (GeneralSecurityException ex) {
			throw new CryptoException(ex);
		}
		return Arrays.equals(msgMac, mac);
	}

	public MacDigester setAlgorithm(String algorithm) {
		this.assertNotInitialized();
		this.algorithm = StringUtils.trimToNull(algorithm);
		return this;
	}

	public MacDigester setProvider(String provider) {
		this.assertNotInitialized();
		this.provider = StringUtils.trimToNull(provider);
		return this;
	}

	public MacDigester setSecret(byte[] secret) {
		this.assertNotInitialized();
		this.secret = ArrayUtils.copy(secret);
		return this;
	}

	public MacDigester setSecret(String secret) {
		AssertUtils.assertNotNull(secret);
		return setSecret(secret.getBytes(CharsetUtils.UTF_8));
	}

	@Override
	public MacDigester copy() {
		MacDigester impl = new MacDigester().setAlgorithm(this.algorithm).setProvider(this.provider);
		impl.secret = ArrayUtils.copy(this.secret);
		return impl;
	}
}
