// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.jwt;

import com.appslandia.common.base.BaseEncoder;
import com.appslandia.common.base.DestroyException;
import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.base.JsonProcessor;
import com.appslandia.common.base.StringWriter;
import com.appslandia.common.crypto.CryptoException;
import com.appslandia.common.crypto.Digester;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.CharsetUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class JwtBuilder extends InitializeObject {

	private JsonProcessor jsonProcessor;
	private Digester digester;

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.jsonProcessor, "jsonProcessor is required.");
		AssertUtils.assertNotNull(this.digester, "digester is required.");
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.digester != null) {
			this.digester.destroy();
		}
	}

	public String build(Object header, Object payload) throws CryptoException {
		this.initialize();

		AssertUtils.assertNotNull(header);
		AssertUtils.assertNotNull(payload);

		StringWriter out = new StringWriter();
		this.jsonProcessor.write(out, header);
		String base64Header = BaseEncoder.BASE64_URL_NP.encode(out.toString().getBytes(CharsetUtils.UTF_8));

		out.reset();
		this.jsonProcessor.write(out, payload);
		String base64Payload = BaseEncoder.BASE64_URL_NP.encode(out.toString().getBytes(CharsetUtils.UTF_8));

		String dataToSign = base64Header + JwtUtils.JWT_PART_SEP + base64Payload;
		String base64Sig = BaseEncoder.BASE64_URL_NP.encode(this.digester.digest(dataToSign.getBytes(CharsetUtils.UTF_8)));
		return dataToSign + JwtUtils.JWT_PART_SEP + base64Sig;
	}

	public boolean verify(String jwt) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(jwt);

		String[] parts = JwtUtils.parseParts(jwt);
		if (parts == null) {
			return false;
		}
		String dataToSign = parts[0] + JwtUtils.JWT_PART_SEP + parts[1];
		byte[] signature = BaseEncoder.BASE64_URL_NP.decode(parts[2]);
		return this.digester.verify(dataToSign.getBytes(CharsetUtils.UTF_8), signature);
	}

	public JwtBuilder setJsonProcessor(JsonProcessor jsonProcessor) {
		assertNotInitialized();
		this.jsonProcessor = jsonProcessor;
		return this;
	}

	public JwtBuilder setDigester(Digester digester) {
		assertNotInitialized();
		this.digester = digester;
		return this;
	}
}