// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.gson;

import java.io.Reader;
import java.io.Writer;

import com.appslandia.common.base.JsonException;
import com.appslandia.common.base.JsonProcessor;
import com.appslandia.common.base.NotBind;
import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class GsonProcessor extends JsonProcessor {

	private Gson gson;
	private GsonBuilder builder;

	@Override
	protected void init() throws Exception {
		if (this.builder == null) {
			this.builder = newGsonBuilder(true);
		}
		this.gson = this.builder.create();
	}

	@Override
	public void write(Writer out, Object obj) throws JsonException {
		this.initialize();
		try {
			this.gson.toJson(obj, out);
		} catch (JsonIOException ex) {
			throw new JsonException(ex);
		}
	}

	@Override
	public <T> T read(Reader reader, Class<? extends T> resultClass) throws JsonException {
		this.initialize();
		try {
			return this.gson.fromJson(reader, resultClass);
		} catch (JsonSyntaxException ex) {
			throw new JsonException(ex);
		} catch (JsonIOException ex) {
			throw new JsonException(ex);
		}
	}

	public GsonProcessor setBuilder(GsonBuilder builder) {
		this.assertNotInitialized();
		this.builder = builder;
		return this;
	}

	@Override
	public GsonProcessor copy() {
		return new GsonProcessor().setBuilder(this.builder);
	}

	public static GsonBuilder newGsonBuilder(boolean serializeNulls) {
		GsonBuilder builder = new GsonBuilder();
		if (serializeNulls) {
			builder.serializeNulls();
		}
		builder.setExclusionStrategies(new ExclusionStrategy() {

			@Override
			public boolean shouldSkipField(FieldAttributes attrs) {
				return attrs.getAnnotation(NotBind.class) != null;
			}

			@Override
			public boolean shouldSkipClass(Class<?> clazz) {
				return false;
			}
		});
		builder.registerTypeAdapter(java.util.Date.class, new DateAdapter());
		builder.registerTypeAdapter(java.sql.Date.class, new SqlDateAdapter());
		builder.registerTypeAdapter(java.sql.Time.class, new SqlTimeAdapter());
		builder.registerTypeAdapter(java.sql.Timestamp.class, new SqlDateTimeAdapter());
		return builder;
	}
}
