// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.Locale;
import java.util.Random;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;

import com.appslandia.common.utils.ArrayUtils;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.RandomUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class PbeEncryptor extends PbeCrypto implements Encryptor {
	private String transformation, provider;

	private String algorithm;
	private boolean initVector;
	private Cipher cipher;

	final Object mutex = new Object();
	final Random random = new SecureRandom();

	@Override
	protected void init() throws Exception {
		super.init();

		// algorithm
		AssertUtils.assertNotNull(this.transformation, "transformation is required.");
		this.algorithm = parseAlgorithm(this.transformation);
		this.initVector = requireIV(this.transformation.toLowerCase(Locale.ENGLISH));

		AssertUtils.assertTrue(this.keySize > 0, "keySize is required.");

		// secretKeyGenerator
		if (this.secretKeyGenerator == null) {
			this.secretKeyGenerator = new SecretKeyGenerator();
		}

		// cipher
		if (this.provider == null) {
			this.cipher = Cipher.getInstance(this.transformation);
		} else {
			this.cipher = Cipher.getInstance(this.transformation, this.provider);
		}
	}

	private static String parseAlgorithm(String spec) {
		int idx = spec.indexOf('/');
		return (idx > 0) ? spec.substring(0, idx) : spec;
	}

	private static boolean requireIV(String spec) {
		return spec.contains("cbc") || spec.contains("cfb") || spec.contains("ofb") || spec.contains("ctr");
	}

	@Override
	public byte[] encrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] iv = null;
		if (this.initVector) {
			iv = RandomUtils.nextBytes(this.random, this.cipher.getBlockSize());
		}

		byte[] salt = RandomUtils.nextBytes(this.random, this.saltSize);
		byte[] encMsg = null;
		SecretKey secretKey = createSecretKey(salt, this.algorithm);

		try {
			synchronized (this.mutex) {
				if (iv != null) {
					this.cipher.init(Cipher.ENCRYPT_MODE, secretKey, new IvParameterSpec(iv));
				} else {
					this.cipher.init(Cipher.ENCRYPT_MODE, secretKey);
				}
				encMsg = this.cipher.doFinal(message);
			}
			if (iv != null) {
				return ArrayUtils.append(iv, salt, encMsg);
			} else {
				return ArrayUtils.append(salt, encMsg);
			}
		} catch (GeneralSecurityException ex) {
			throw new CryptoException(ex);
		} finally {
			CryptoUtils.tryDestroy(secretKey);
		}
	}

	@Override
	public byte[] decrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] iv = null;
		int msgLen = message.length;

		if (this.initVector) {
			AssertUtils.assertTrue(msgLen > this.cipher.getBlockSize() + this.saltSize, "message is invalid.");
			msgLen = msgLen - this.cipher.getBlockSize() - this.saltSize;
			iv = new byte[this.cipher.getBlockSize()];
		} else {
			AssertUtils.assertTrue(msgLen > this.saltSize, "message is invalid.");
			msgLen = msgLen - this.saltSize;
		}

		byte[] salt = new byte[this.saltSize];
		byte[] msg = new byte[msgLen];

		if (iv != null) {
			ArrayUtils.copy(message, iv, salt, msg);
		} else {
			ArrayUtils.copy(message, salt, msg);
		}

		SecretKey secretKey = createSecretKey(salt, this.algorithm);
		try {
			synchronized (this.mutex) {
				if (iv != null) {
					this.cipher.init(Cipher.DECRYPT_MODE, secretKey, new IvParameterSpec(iv));
				} else {
					this.cipher.init(Cipher.DECRYPT_MODE, secretKey);
				}
				return this.cipher.doFinal(msg);
			}
		} catch (GeneralSecurityException ex) {
			throw new CryptoException(ex);
		} finally {
			CryptoUtils.tryDestroy(secretKey);
		}
	}

	public PbeEncryptor setTransformation(String transformation) {
		this.assertNotInitialized();
		this.transformation = StringUtils.trimToNull(transformation);
		return this;
	}

	public PbeEncryptor setProvider(String provider) {
		this.assertNotInitialized();
		this.provider = StringUtils.trimToNull(provider);
		return this;
	}

	@Override
	public PbeEncryptor setSaltSize(int saltSize) {
		super.setSaltSize(saltSize);
		return this;
	}

	@Override
	public PbeEncryptor setIterationCount(int iterationCount) {
		super.setIterationCount(iterationCount);
		return this;
	}

	@Override
	public PbeEncryptor setKeySize(int keySize) {
		super.setKeySize(keySize);
		return this;
	}

	@Override
	public PbeEncryptor setSecurePassword(SecureString password) {
		super.setSecurePassword(password);
		return this;
	}

	@Override
	public PbeEncryptor setSecurePassword(char[] password) {
		super.setSecurePassword(password);
		return this;
	}

	@Override
	public PbeEncryptor setPassword(char[] password) {
		super.setPassword(password);
		return this;
	}

	@Override
	public PbeEncryptor setSecretKeyGenerator(SecretKeyGenerator secretKeyGenerator) {
		super.setSecretKeyGenerator(secretKeyGenerator);
		return this;
	}

	@Override
	public PbeEncryptor copy() {
		PbeEncryptor impl = new PbeEncryptor().setTransformation(this.transformation).setProvider(this.provider);
		impl.setSaltSize(this.saltSize).setIterationCount(this.iterationCount).setKeySize(this.keySize);

		if (this.password != null) {
			if (this.password instanceof SecureString) {
				impl.password = ((SecureString) this.password).copy();
			} else {
				impl.setPassword((char[]) this.password);
			}
		}
		if (this.secretKeyGenerator != null) {
			impl.secretKeyGenerator = this.secretKeyGenerator.copy();
		}
		return impl;
	}
}
