// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import com.appslandia.common.base.DestroyException;
import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.ArrayUtils;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.MathUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class DsaSigner extends InitializeObject implements Encryptor {
	private DsaDigester signer;

	public DsaSigner() {
	}

	public DsaSigner(DsaDigester signer) {
		this.signer = signer;
	}

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.signer, "signer is required.");
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.signer != null) {
			this.signer.destroy();
		}
		super.destroy();
	}

	@Override
	public byte[] encrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] digested = this.signer.digest(message);
		byte[] len = MathUtils.toByteArray(digested.length);
		return ArrayUtils.append(len, digested, message);
	}

	@Override
	public byte[] decrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] len = new byte[4];
		AssertUtils.assertTrue(message.length > 4, "message is invalid.");
		ArrayUtils.copy(message, len);

		int digestLength = MathUtils.toInt(len);
		AssertUtils.assertTrue(digestLength > 0, "message is invalid.");

		byte[] digested = new byte[digestLength];
		byte[] parsedMsg = new byte[message.length - digestLength - 4];
		System.arraycopy(message, 4, digested, 0, digestLength);
		System.arraycopy(message, 4 + digestLength, parsedMsg, 0, parsedMsg.length);

		if (!this.signer.verify(parsedMsg, digested)) {
			throw new CryptoException("message was tampered.");
		}
		return parsedMsg;
	}

	public DsaSigner setSigner(DsaDigester signer) {
		assertNotInitialized();
		this.signer = signer;
		return this;
	}

	@Override
	public DsaSigner copy() {
		DsaSigner impl = new DsaSigner();
		if (this.signer != null) {
			impl.signer = this.signer.copy();
		}
		return impl;
	}
}
