// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.cdi;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.Set;

import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.enterprise.inject.spi.InjectionTarget;
import javax.enterprise.inject.spi.PassivationCapable;

import com.appslandia.common.utils.ArrayUtils;
import com.appslandia.common.utils.CollectionUtils;
import com.appslandia.common.utils.ObjectUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class BeanImpl<T> implements Bean<T>, PassivationCapable {

	final BeanDefinition<T> definition;
	final InjectionTarget<T> it;
	final Set<Annotation> qualifiers;
	final Set<Type> types;

	public BeanImpl(BeanDefinition<T> definition, BeanManager beanManager) {
		this.definition = definition;

		AnnotatedType<T> at = beanManager.createAnnotatedType(ObjectUtils.cast(definition.beanClass));
		this.it = beanManager.createInjectionTarget(at);

		this.qualifiers = ArrayUtils.hasElements(definition.qualifiers) ? CollectionUtils.unmodifiableSet(definition.qualifiers) : Collections.emptySet();
		this.types = CollectionUtils.unmodifiableSet(definition.types);
	}

	@Override
	public Set<Type> getTypes() {
		return this.types;
	}

	@Override
	public Set<Annotation> getQualifiers() {
		return this.qualifiers;
	}

	@Override
	public Class<? extends Annotation> getScope() {
		return this.definition.scope;
	}

	@Override
	public String getName() {
		return this.definition.name;
	}

	@Override
	public Set<Class<? extends Annotation>> getStereotypes() {
		return Collections.emptySet();
	}

	@Override
	public boolean isAlternative() {
		return false;
	}

	@Override
	public Class<?> getBeanClass() {
		return this.definition.beanClass;
	}

	@Override
	public Set<InjectionPoint> getInjectionPoints() {
		return this.it.getInjectionPoints();
	}

	@Override
	public boolean isNullable() {
		return false;
	}

	@Override
	public T create(CreationalContext<T> ctx) {
		T t = this.it.produce(ctx);
		this.it.inject(t, ctx);
		if (this.definition.consumer != null) {
			this.definition.consumer.accept(ObjectUtils.cast(t));
		}
		this.it.postConstruct(t);
		return t;
	}

	@Override
	public void destroy(T t, CreationalContext<T> ctx) {
		this.it.preDestroy(t);
		this.it.dispose(t);
		ctx.release();
	}

	@Override
	public String getId() {
		return this.definition.passivationCapableId;
	}
}
