// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.security.ProtectionDomain;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.UUID;

import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.ExceptionUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.ReflectionUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ToStringBuilder {

	@Target({ ElementType.TYPE, ElementType.FIELD, ElementType.METHOD })
	@Retention(RetentionPolicy.RUNTIME)
	@Documented
	public @interface EnableDecision {
		Decision value();
	}

	public abstract static class MemberDecision {

		static boolean isJdkClass(Class<?> type) {
			// @formatter:off
			return	type.getName().startsWith("java.") || 
					type.getName().startsWith("javax.") || 
					type.getName().startsWith("sun.") ||
					type.getName().startsWith("com.sun.") || 
					type.getName().startsWith("com.oracle.") || 
					type.getName().startsWith("jdk.") ||
					type.getName().startsWith("org.omg.") || 
					type.getName().startsWith("org.w3c.");
			// @formatter:on
		}

		public Decision getDecision(Object value, Member member) {
			EnableDecision decision = null;
			if (member != null) {
				Class<?> type = (member instanceof Field) ? ((Field) member).getType() : ((Method) member).getReturnType();
				if (!isJdkClass(type)) {
					if (member instanceof Field) {
						decision = ((Field) member).getAnnotation(EnableDecision.class);
					} else {
						decision = ((Method) member).getAnnotation(EnableDecision.class);
					}
				}
			}
			if (decision == null) {
				if (!isJdkClass(value.getClass())) {
					decision = ReflectionUtils.findAnnotation(value.getClass(), EnableDecision.class);
				}
			}
			if (decision != null) {
				return decision.value();
			}
			return doGetDecision(value, member);
		}

		protected abstract Decision doGetDecision(Object value, Member member);
	}

	public static final MemberDecision DEFAULT_DECISION = new MemberDecision() {

		@Override
		protected Decision doGetDecision(Object value, Member member) {
			return Decision.TO_STRING;
		}
	};

	public static enum Decision {
		TO_STRING, OBJECT_INFO, TYPE_INFO
	}

	private int level;
	private int identTabs;
	private boolean singleLine;
	private MemberDecision memberDecision = DEFAULT_DECISION;

	public ToStringBuilder() {
		this(2);
	}

	public ToStringBuilder(int level) {
		setLevel(level);
	}

	public ToStringBuilder memberDecision(MemberDecision memberDecision) {
		this.memberDecision = memberDecision;
		return this;
	}

	public String toString(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs);
		if (obj == null) {
			return builder.append("null").toString();
		}
		this.toStringObject(obj, 1, builder);
		return builder.toString();
	}

	public String toStringFields(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs);
		if (obj == null) {
			return builder.append("null").toString();
		}
		this.toStringFields(obj, 1, builder);
		return builder.toString();
	}

	private void toStringObject(Object obj, int level, TextBuilder builder) {
		if (obj == null) {
			builder.append("null");
			return;
		}
		if (obj instanceof Iterable) {
			toStringIterator(obj, new IteratorIterator(((Iterable<?>) obj).iterator()), level, builder);
			return;
		}
		if (obj instanceof Iterator) {
			toStringIterator(obj, new IteratorIterator((Iterator<?>) obj), level, builder);
			return;
		}
		if (obj instanceof Enumeration) {
			toStringIterator(obj, new EnumerationIterator((Enumeration<?>) obj), level, builder);
			return;
		}
		if (obj.getClass() == byte[].class) {
			builder.append(ObjectUtils.toObjectInfo(obj)).append(" (").append(((byte[]) obj).length).append(")");
			return;
		}
		if (obj.getClass() == char[].class) {
			builder.append(ObjectUtils.toObjectInfo(obj)).append(" (").append(((char[]) obj).length).append(")");
			return;
		}
		if (obj.getClass().isArray()) {
			toStringIterator(obj, new ArrayIterator(obj), level, builder);
			return;
		}
		if (obj instanceof Map) {
			toStringMap((Map<?, ?>) obj, level, builder);
			return;
		}
		if (obj instanceof Throwable) {
			builder.append(obj.getClass().getName()).append("(").append(ExceptionUtils.toCausePath((Throwable) obj)).append(")");
			return;
		}
		if (obj.getClass() == String.class) {
			builder.append("\"").append(obj).append("\"").append(" (").append(((String) obj).length()).append(")");
			return;
		}
		if (useToString(obj)) {
			builder.append(obj);
			return;
		}
		if (useTypeAndToString(obj)) {
			builder.append(obj.getClass().getName()).append("(").append(obj).append(")");
			return;
		}
		if (obj instanceof TimeZone) {
			builder.append(obj.getClass().getName()).append("(").append(((TimeZone) obj).getID()).append(")");
			return;
		}
		if (obj instanceof Calendar) {
			Calendar cal = (Calendar) obj;
			builder.append(cal.getClass().getName()).append("(date/time=").append(cal.getTime()).append(", zone=").append(cal.getTimeZone().getID()).append(")");
			return;
		}
		// Others
		toStringFields(obj, level, builder);
	}

	private void toStringFields(Object obj, int level, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(obj));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		// Fields
		Class<?> searchType = obj.getClass();
		while (searchType != null) {
			Field[] fields = searchType.getDeclaredFields();
			for (Field field : fields) {
				if (!isFirst) {
					builder.append(",");
				} else {
					isFirst = false;
				}

				appendln(builder);
				appendtab(builder, level + this.identTabs);
				builder.append(field.getName()).append(": ");

				try {
					field.setAccessible(true);
					Object fieldVal = field.get(obj);
					if (fieldVal == null) {
						builder.append("null");
					} else {
						Decision decision = this.memberDecision.getDecision(fieldVal, field);
						if ((decision == null) || (decision == Decision.TO_STRING)) {
							this.toStringObject(fieldVal, level + 1, builder);

						} else if (decision == Decision.OBJECT_INFO) {
							builder.append(ObjectUtils.toObjectInfo(fieldVal));
						} else {
							builder.append(fieldVal.getClass());
						}
					}

				} catch (Throwable ex) {
					builder.append("error=").append(ExceptionUtils.toCausePath(ex));
				}
			}
			searchType = searchType.getSuperclass();
		}

		// Getters
		try {
			for (Method method : obj.getClass().getMethods()) {
				if (method.getName().equals("getClass")) {
					continue;
				}
				// Parse field
				String fieldName = parseField(method);
				if (fieldName == null) {
					continue;
				}
				if (ReflectionUtils.findField(obj.getClass(), fieldName) != null) {
					continue;
				}
				if (!isFirst) {
					builder.append(",");
				} else {
					isFirst = false;
				}

				appendln(builder);
				appendtab(builder, level + this.identTabs);
				builder.append(method.getName()).append("(): ");

				try {
					Object mthVal = method.invoke(obj);
					if (mthVal == null) {
						builder.append("null");
					} else {
						Decision decision = this.memberDecision.getDecision(mthVal, method);
						if ((decision == null) || (decision == Decision.TO_STRING)) {
							this.toStringObject(mthVal, level + 1, builder);

						} else if (decision == Decision.OBJECT_INFO) {
							builder.append(ObjectUtils.toObjectInfo(mthVal));
						} else {
							builder.append(mthVal.getClass());
						}
					}

				} catch (Throwable ex) {
					builder.append("error=").append(ExceptionUtils.toCausePath(ex));
				}
			}
		} catch (Throwable ex) {
			builder.append("error=").append(ExceptionUtils.toCausePath(ex));
		}

		if (isFirst) {
			builder.append(" no fields/getters ]");
		} else {
			appendln(builder);
			appendtab(builder, level - 1 + this.identTabs).append("]");
		}
	}

	private void toStringIterator(Object obj, ElementIterator iterator, int level, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(obj));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		while (iterator.hasNext()) {
			Object element = iterator.next();

			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder);
			appendtab(builder, level + this.identTabs);

			if (element == null) {
				builder.append("null");
			} else {
				Decision decision = this.memberDecision.getDecision(element, null);
				if ((decision == null) || (decision == Decision.TO_STRING)) {
					this.toStringObject(element, level + 1, builder);

				} else if (decision == Decision.OBJECT_INFO) {
					builder.append(ObjectUtils.toObjectInfo(element));
				} else {
					builder.append(element.getClass());
				}
			}
		}
		if (isFirst) {
			builder.append(" no elements ]");
		} else {
			appendln(builder);
			appendtab(builder, level - 1 + this.identTabs).append("] (").append(iterator.getIndex()).append(")");
		}
	}

	private void toStringMap(Map<?, ?> map, int level, TextBuilder builder) {
		builder.append(ObjectUtils.toObjectInfo(map));
		if (level > this.level) {
			return;
		}
		builder.append("[");
		boolean isFirst = true;

		for (Object key : map.keySet()) {
			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder);
			appendtab(builder, level + this.identTabs);

			builder.append(key).append(": ");
			Object entryVal = map.get(key);
			if (entryVal == null) {
				builder.append("null");
			} else {
				Decision decision = this.memberDecision.getDecision(entryVal, null);
				if ((decision == null) || (decision == Decision.TO_STRING)) {
					this.toStringObject(entryVal, level + 1, builder);

				} else if (decision == Decision.OBJECT_INFO) {
					builder.append(ObjectUtils.toObjectInfo(entryVal));
				} else {
					builder.append(entryVal.getClass());
				}
			}
		}

		if (isFirst) {
			builder.append(" no entries ]");
		} else {
			appendln(builder);
			appendtab(builder, level - 1 + this.identTabs).append("] (").append(map.size()).append(")");
		}
	}

	private void toStringAttributes(Object obj, Method getAttributeMethod, Set<String> attributes, int level, TextBuilder builder) {
		builder.append("[");
		boolean isFirst = true;

		for (String attribute : attributes) {
			if (!isFirst) {
				builder.append(",");
			} else {
				isFirst = false;
			}
			appendln(builder);
			appendtab(builder, level + this.identTabs);
			builder.append(attribute).append(": ");

			try {
				Object element = getAttributeMethod.invoke(obj, attribute);
				if (element == null) {
					builder.append("null");
				} else {
					Decision decision = this.memberDecision.getDecision(element, null);
					if ((decision == null) || (decision == Decision.TO_STRING)) {
						this.toStringObject(element, level + 1, builder);

					} else if (decision == Decision.OBJECT_INFO) {
						builder.append(ObjectUtils.toObjectInfo(element));
					} else {
						builder.append(element.getClass());
					}
				}

			} catch (Throwable ex) {
				builder.append("error=").append(ExceptionUtils.toCausePath(ex));
			}
		}

		if (isFirst) {
			builder.append(" no elements ]");
		} else {
			appendln(builder);
			appendtab(builder, level - 1 + this.identTabs).append("]");
		}
	}

	public String toStringAttributes(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs);
		if (obj == null) {
			builder.append("null");
			return builder.toString();
		}
		try {
			Set<String> attributes = getAttributeNames(obj, "getAttributeNames");
			Method method = ReflectionUtils.findMethod(obj.getClass(), "getAttribute");
			AssertUtils.assertNotNull(method);

			builder.append(ObjectUtils.toObjectInfo(obj)).append("-attributes");
			toStringAttributes(obj, method, attributes, 1, builder);
		} catch (Throwable ex) {
			builder.append("error=").append(ExceptionUtils.toCausePath(ex));
		}
		return builder.toString();
	}

	public String toStringHeaders(Object obj) {
		TextBuilder builder = new TextBuilder();
		appendtab(builder, this.identTabs);
		if (obj == null) {
			builder.append("null");
			return builder.toString();
		}
		try {
			Set<String> attributes = getAttributeNames(obj, "getHeaderNames");
			Method method = ReflectionUtils.findMethod(obj.getClass(), "getHeaders");
			AssertUtils.assertNotNull(method);

			builder.append(ObjectUtils.toObjectInfo(obj)).append("-headers");
			toStringAttributes(obj, method, attributes, 1, builder);
		} catch (Throwable ex) {
			builder.append("error=").append(ExceptionUtils.toCausePath(ex));
		}
		return builder.toString();
	}

	private static Set<String> getAttributeNames(Object obj, String methodName) throws Exception {
		Method method = ReflectionUtils.findMethod(obj.getClass(), methodName);
		AssertUtils.assertNotNull(method);

		Object attrs = method.invoke(obj);
		Set<String> names = new TreeSet<>();
		if (attrs instanceof Enumeration) {

			Enumeration<String> enm = ObjectUtils.cast(attrs);
			while (enm.hasMoreElements()) {
				names.add(enm.nextElement());
			}
		} else {
			Collection<String> attrCol = ObjectUtils.cast(attrs);
			names.addAll(attrCol);
		}
		return names;
	}

	public ToStringBuilder setLevel(int level) {
		this.level = Math.max(level, 1);
		return this;
	}

	public ToStringBuilder setIdentTabs(int identTabs) {
		this.identTabs = Math.max(identTabs, 0);
		return this;
	}

	public ToStringBuilder setSingleLine(boolean singleLine) {
		this.singleLine = singleLine;
		return this;
	}

	private TextBuilder appendln(TextBuilder builder) {
		if (!this.singleLine) {
			builder.appendln();
		}
		return builder;
	}

	private TextBuilder appendtab(TextBuilder builder, int tabs) {
		if (this.singleLine) {
			builder.appendsp();
		} else {
			builder.appendtab(tabs);
		}
		return builder;
	}

	static boolean useToString(Object obj) {
		if (obj instanceof Number || obj.getClass() == Boolean.class || obj.getClass() == Character.class || obj instanceof CharSequence) {
			return true;
		}
		if (obj instanceof Date || obj.getClass().getName().startsWith("java.time.")) {
			return true;
		}
		if (obj.getClass().getName().startsWith("java.util.concurrent.atomic.")) {
			return true;
		}

		if (obj instanceof File || obj.getClass() == URL.class || obj.getClass() == URI.class || obj.getClass() == Void.class) {
			return true;
		}
		if (obj.getClass() == Locale.class || obj instanceof Charset || obj instanceof Enum || obj.getClass() == UUID.class) {
			return true;
		}
		if (obj.getClass() == TextBuilder.class) {
			return true;
		}
		if (obj instanceof Type || obj instanceof Package || obj instanceof ProtectionDomain) {
			return true;
		}
		return false;
	}

	static boolean useTypeAndToString(Object obj) {
		if (obj instanceof ResourceBundle) {
			return true;
		}
		if (obj instanceof Reader || obj instanceof Writer || obj instanceof InputStream || obj instanceof OutputStream) {
			return true;
		}
		if (obj instanceof ByteBuffer || obj instanceof CharBuffer || obj instanceof ByteChunks) {
			return true;
		}
		if (obj instanceof ClassLoader) {
			return true;
		}
		return false;
	}

	static String parseField(Method mth) {
		// JDK8: mth.getParameterCount() > 0
		if (mth.getParameterTypes().length > 0) {
			return null;
		}
		if (mth.getReturnType() == void.class) {
			return null;
		}
		if (mth.getName().equals("get") || mth.getName().equals("is")) {
			return null;
		}
		if (mth.getName().startsWith("get")) {
			return StringUtils.firstLowerCase(mth.getName().substring(3));
		}
		if (mth.getName().startsWith("is")) {
			return StringUtils.firstLowerCase(mth.getName().substring(2));
		}
		return null;
	}

	interface ElementIterator {

		boolean hasNext();

		Object next();

		int getIndex();
	}

	static class ArrayIterator implements ElementIterator {
		final Object obj;
		final int len;
		int index = 0;

		public ArrayIterator(Object obj) {
			this.obj = obj;
			this.len = Array.getLength(obj);
		}

		@Override
		public boolean hasNext() {
			return this.index < this.len;
		}

		@Override
		public Object next() {
			return Array.get(this.obj, this.index++);
		}

		@Override
		public int getIndex() {
			return this.index;
		}
	}

	static class IteratorIterator implements ElementIterator {
		final Iterator<?> obj;
		int index = 0;

		public IteratorIterator(Iterator<?> obj) {
			this.obj = obj;
		}

		@Override
		public boolean hasNext() {
			return this.obj.hasNext();
		}

		@Override
		public Object next() {
			this.index++;
			return this.obj.next();
		}

		@Override
		public int getIndex() {
			return this.index;
		}
	}

	static class EnumerationIterator implements ElementIterator {
		final Enumeration<?> obj;
		int index = 0;

		public EnumerationIterator(Enumeration<?> obj) {
			this.obj = obj;
		}

		@Override
		public boolean hasNext() {
			return this.obj.hasMoreElements();
		}

		@Override
		public Object next() {
			this.index++;
			return this.obj.nextElement();
		}

		@Override
		public int getIndex() {
			return this.index;
		}
	}
}
