// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import com.appslandia.common.base.ConfigBase;
import com.appslandia.common.base.FormatProvider;
import com.appslandia.common.base.FormatProviderImpl;
import com.appslandia.common.formatters.Formatter;
import com.appslandia.common.formatters.FormatterException;
import com.appslandia.common.formatters.FormatterProvider;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class PropertyUtils {

	private static interface PropertyStrategy {

		Method find(Class<?> clazz, String propertyName);
	}

	public static final PropertyStrategy METHOD_PROPERTY_STRATEGY = new PropertyStrategy() {

		@Override
		public Method find(Class<?> clazz, final String propertyName) {
			return ReflectionUtils.traverse(clazz, new ReflectionUtils.MethodMatcher() {

				@Override
				public boolean match(Method m) {
					if (!Modifier.isPublic(m.getModifiers()) || Modifier.isStatic(m.getModifiers())) {
						return false;
					}
					// JDK8: m.getParameterCount()
					if (m.getParameterTypes().length != 1) {
						return false;
					}
					return m.getName().equals(propertyName);
				}
			}, ReflectionUtils.FIND_METHOD_TRAVERSER);
		}
	};

	public static final PropertyStrategy BEAN_PROPERTY_STRATEGY = new PropertyStrategy() {

		@Override
		public Method find(Class<?> clazz, final String propertyName) {
			return ReflectionUtils.traverse(clazz, new ReflectionUtils.MethodMatcher() {

				@Override
				public boolean match(Method m) {
					if (!Modifier.isPublic(m.getModifiers()) || Modifier.isStatic(m.getModifiers())) {
						return false;
					}
					// JDK8: m.getParameterCount()
					if (m.getParameterTypes().length != 1) {
						return false;
					}
					return m.getName().equals("set" + StringUtils.firstUpperCase(propertyName));
				}
			}, ReflectionUtils.FIND_METHOD_TRAVERSER);
		}
	};

	public static void initialize(Object obj, ConfigBase props, PropertyStrategy propertyStrategy) throws FormatterException, ReflectionException {
		initialize(obj, props, new FormatProviderImpl(), propertyStrategy);
	}

	public static void initialize(Object obj, ConfigBase props, FormatProvider formatProvider, PropertyStrategy propertyStrategy) throws FormatterException, ReflectionException {
		for (String prop : props.getKeys()) {
			Method m = propertyStrategy.find(obj.getClass(), prop);
			if (m == null) {
				continue;
			}
			Class<?> parameterType = m.getParameterTypes()[0];
			if (parameterType == String.class) {
				ReflectionUtils.invoke(m, obj, props.getString(prop));
			} else {
				Formatter formatter = FormatterProvider.getDefault().getFormatter(parameterType);
				Object value = formatter.parse(props.getString(prop), formatProvider);
				ReflectionUtils.invoke(m, obj, value);
			}
		}
	}
}
